<?php

namespace App\Notifications;

use App\Models\User;
use App\Models\Product;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class StockTransferredNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $product;
    public $quantityTransferred;
    public $fromUser; // Anayetuma
    public $toUser;   // Anayepokea

    /**
     * Create a new notification instance.
     */
    public function __construct(Product $product, $quantityTransferred, User $fromUser, User $toUser)
    {
        $this->product = $product;
        $this->quantityTransferred = $quantityTransferred;
        $this->fromUser = $fromUser;
        $this->toUser = $toUser;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
                    ->subject('Stock Transfer Initiated')
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("You have successfully transferred **{$this->quantityTransferred} {$this->product->unit}** of **{$this->product->name}** to **{$this->toUser->name}** ({$this->toUser->role}).")
                    ->line("Remaining stock of {$this->product->name} in your account: **{$this->product->quantity} {$this->product->unit}**.")
                    ->action('View Stock Outputs', url('/stock-outputs')) // Au route yako ya stock outputs
                    ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification for database storage.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'stock_transferred_by_you',
            'product_id' => $this->product->id,
            'product_name' => $this->product->name,
            'quantity_transferred' => $this->quantityTransferred,
            'from_user_id' => $this->fromUser->id,
            'from_user_name' => $this->fromUser->name,
            'to_user_id' => $this->toUser->id,
            'to_user_name' => $this->toUser->name,
            'message' => "You transferred {$this->quantityTransferred} {$this->product->unit} of '{$this->product->name}' to {$this->toUser->name}.",
            'link' => url('/stock-outputs'),
        ];
    }
}