<?php

namespace App\Notifications;

use App\Models\User;
use App\Models\Product;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class StockReceivedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $product;
    public $quantityReceived;
    public $fromUser; // Anayetuma
    public $toUser;   // Anayepokea

    /**
     * Create a new notification instance.
     */
    public function __construct(Product $product, $quantityReceived, User $fromUser, User $toUser)
    {
        $this->product = $product;
        $this->quantityReceived = $quantityReceived;
        $this->fromUser = $fromUser;
        $this->toUser = $toUser;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
                    ->subject('Stock Received Confirmation')
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("You have received **{$this->quantityReceived} {$this->product->unit}** of **{$this->product->name}** from **{$this->fromUser->name}** ({$this->fromUser->role}).")
                    ->line("Your current stock of {$this->product->name} is now: **{$this->product->quantity} {$this->product->unit}**.")
                    ->action('View Your Stock', url('/products')) // Au route yako ya kuonyesha stock za user
                    ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification for database storage.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'stock_received_by_you',
            'product_id' => $this->product->id,
            'product_name' => $this->product->name,
            'quantity_received' => $this->quantityReceived,
            'from_user_id' => $this->fromUser->id,
            'from_user_name' => $this->fromUser->name,
            'to_user_id' => $this->toUser->id,
            'to_user_name' => $this->toUser->name,
            'message' => "You received {$this->quantityReceived} {$this->product->unit} of '{$this->product->name}' from {$this->fromUser->name}.",
            'link' => url('/products'),
        ];
    }
}