<?php

namespace App\Notifications;

use App\Models\User;
use App\Models\Product;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class StockAddedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $product;
    public $quantityAdded;
    public $addedByUser; // User (Admin/Storekeeper) aliyeongeza stock
    public $isNewProduct; // Boolean kuashiria kama ni bidhaa mpya au stock iliyoongezwa

    /**
     * Create a new notification instance.
     */
    public function __construct(Product $product, $quantityAdded, User $addedByUser, $isNewProduct = false)
    {
        $this->product = $product;
        $this->quantityAdded = $quantityAdded;
        $this->addedByUser = $addedByUser;
        $this->isNewProduct = $isNewProduct;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $subject = $this->isNewProduct ? 'New Product & Stock Added' : 'Stock Added to Existing Product';
        $messageLine = $this->isNewProduct
            ? "**{$this->product->name}** with a quantity of **{$this->quantityAdded} {$this->product->unit}** has been registered and stocked by **{$this->addedByUser->name}** ({$this->addedByUser->role})."
            : "A quantity of **{$this->quantityAdded} {$this->product->unit}** has been added to **{$this->product->name}** by **{$this->addedByUser->name}** ({$this->addedByUser->role}).";

        return (new MailMessage)
                    ->subject($subject)
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line($messageLine)
                    ->line("Current total stock for {$this->product->name}: **{$this->product->quantity} {$this->product->unit}**.")
                    ->action('View Product', url('/products/' . $this->product->id))
                    ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification for database storage.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        $message = $this->isNewProduct
            ? "New product '{$this->product->name}' (Qty: {$this->quantityAdded} {$this->product->unit}) registered by {$this->addedByUser->name}."
            : "Stock (Qty: {$this->quantityAdded} {$this->product->unit}) added to '{$this->product->name}' by {$this->addedByUser->name}.";

        return [
            'type' => $this->isNewProduct ? 'product_added' : 'stock_added',
            'product_id' => $this->product->id,
            'product_name' => $this->product->name,
            'quantity_added' => $this->quantityAdded,
            'current_quantity' => $this->product->quantity,
            'added_by_user_id' => $this->addedByUser->id,
            'added_by_user_name' => $this->addedByUser->name,
            'message' => $message,
            'link' => url('/products/' . $this->product->id),
        ];
    }
}