<?php

namespace App\Notifications;

use App\Models\User;
use App\Models\Sale;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class SaleRecordedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $sale;
    public $recordedByUser;

    /**
     * Create a new notification instance.
     */
    public function __construct(Sale $sale, User $recordedByUser)
    {
        $this->sale = $sale;
        $this->recordedByUser = $recordedByUser;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $productName = $this->sale->product->name ?? 'Unknown Product';
        $customerName = $this->sale->customer->username ?? 'Unknown Customer';

        return (new MailMessage)
                    ->subject('New Sale Recorded')
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("A new sale has been recorded in the system by **{$this->recordedByUser->name}** ({$this->recordedByUser->role}).")
                    ->line("Details:")
                    ->line("- **Product:** {$productName}")
                    ->line("- **Quantity:** {$this->sale->quantity} {$this->sale->product->unit}")
                    ->line("- **Total Price:** TZS " . number_format($this->sale->total_price, 2))
                    ->line("- **Customer:** {$customerName}")
                    ->action('View Sale', url('/sales/' . $this->sale->id))
                    ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification for database storage.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        $productName = $this->sale->product->name ?? 'Unknown Product';
        $customerName = $this->sale->customer->username ?? 'Unknown Customer';

        $message = "New sale: {$this->sale->quantity} {$this->sale->product->unit} of '{$productName}' for TZS " . number_format($this->sale->total_price, 2) . " sold to '{$customerName}' by {$this->recordedByUser->name}.";

        return [
            'type' => 'sale_recorded',
            'sale_id' => $this->sale->id,
            'product_name' => $productName,
            'quantity' => $this->sale->quantity,
            'total_price' => $this->sale->total_price,
            'customer_name' => $customerName,
            'recorded_by_user_id' => $this->recordedByUser->id,
            'recorded_by_user_name' => $this->recordedByUser->name,
            'message' => $message,
            'link' => url('/sales/' . $this->sale->id),
        ];
    }
}