<?php

namespace App\Notifications;

use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class SaleDeletedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $saleDetails; // String yenye maelezo ya sale iliyofutwa
    public $deletedByUser;

    /**
     * Create a new notification instance.
     */
    public function __construct(string $saleDetails, User $deletedByUser)
    {
        $this->saleDetails = $saleDetails;
        $this->deletedByUser = $deletedByUser;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
                    ->subject('Sale Record Deleted')
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("A sale record has been deleted from the system by **{$this->deletedByUser->name}** ({$this->deletedByUser->role}).")
                    ->line("Details of the deleted sale: **{$this->saleDetails}**")
                    ->line('Please review your sales records. The stock quantity has been reverted.')
                    ->action('View All Sales', url('/sales'))
                    ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification for database storage.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'sale_deleted',
            'sale_details' => $this->saleDetails,
            'deleted_by_user_id' => $this->deletedByUser->id,
            'deleted_by_user_name' => $this->deletedByUser->name,
            'message' => "Sale '{$this->saleDetails}' deleted by {$this->deletedByUser->name}. Stock reverted.",
            'link' => url('/sales'),
        ];
    }
}