<?php

namespace App\Notifications;

use App\Models\User;
use App\Models\SalaryPayment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class SalaryPaidNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $salaryPayment;
    public $paidByUser; // Admin aliyefanya malipo

    /**
     * Create a new notification instance.
     */
    public function __construct(SalaryPayment $salaryPayment, User $paidByUser)
    {
        $this->salaryPayment = $salaryPayment;
        $this->paidByUser = $paidByUser;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        // Admin anayepokea noti, atapata kupitia mail na database
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $recipientUser = $this->salaryPayment->user; // User aliyelipwa mshahara

        return (new MailMessage)
                    ->subject('Salary Payment Recorded')
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("A salary payment of **TZS " . number_format($this->salaryPayment->amount, 2) . "** has been recorded for **{$recipientUser->name}** ({$recipientUser->email})")
                    ->line("for the period **{$this->salaryPayment->payment_period}** on **{$this->salaryPayment->payment_date->format('Y-m-d')}** by **{$this->paidByUser->name}**.")
                    ->action('View Salary Payments', url('/salaries'))
                    ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification for database storage.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        $recipientUser = $this->salaryPayment->user;
        $message = "Salary of TZS " . number_format($this->salaryPayment->amount, 2) . " for '{$recipientUser->name}' ({$this->salaryPayment->payment_period}) recorded by {$this->paidByUser->name}.";

        return [
            'type' => 'salary_paid',
            'salary_payment_id' => $this->salaryPayment->id,
            'user_paid_id' => $recipientUser->id,
            'user_paid_name' => $recipientUser->name,
            'amount' => $this->salaryPayment->amount,
            'payment_date' => $this->salaryPayment->payment_date,
            'payment_period' => $this->salaryPayment->payment_period,
            'recorded_by_user_id' => $this->paidByUser->id,
            'recorded_by_user_name' => $this->paidByUser->name,
            'message' => $message,
            'link' => url('/salaries/' . $this->salaryPayment->id),
        ];
    }
}