<?php

namespace App\Notifications;

use App\Models\User;
use App\Models\OtherIncome;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class OtherIncomeRecordedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $otherIncome;
    public $recordedByUser;

    /**
     * Create a new notification instance.
     */
    public function __construct(OtherIncome $otherIncome, User $recordedByUser)
    {
        $this->otherIncome = $otherIncome;
        $this->recordedByUser = $recordedByUser;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
                    ->subject('New Other Income Recorded')
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("A new other income has been recorded in the system by **{$this->recordedByUser->name}**.")
                    ->line("Details:")
                    ->line("- **Source:** {$this->otherIncome->source}")
                    ->line("- **Amount:** TZS " . number_format($this->otherIncome->amount, 2))
                    ->line("- **Date:** {$this->otherIncome->income_date->format('Y-m-d')}")
                    ->lineIf($this->otherIncome->description, "- **Description:** {$this->otherIncome->description}")
                    ->action('View Other Income', url('/other-incomes/' . $this->otherIncome->id))
                    ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification for database storage.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        $message = "New other income of TZS " . number_format($this->otherIncome->amount, 2) . " from '{$this->otherIncome->source}' recorded by {$this->recordedByUser->name}.";
        if ($this->otherIncome->description) {
            $message .= " (Description: {$this->otherIncome->description})";
        }

        return [
            'type' => 'other_income_recorded',
            'income_id' => $this->otherIncome->id,
            'source' => $this->otherIncome->source,
            'amount' => $this->otherIncome->amount,
            'recorded_by_user_id' => $this->recordedByUser->id,
            'recorded_by_user_name' => $this->recordedByUser->name,
            'message' => $message,
            'link' => url('/other-incomes/' . $this->otherIncome->id),
        ];
    }
}