<?php

namespace App\Notifications;

use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class OtherIncomeDeletedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $incomeDetails; // String yenye maelezo ya income iliyofutwa
    public $deletedByUser;

    /**
     * Create a new notification instance.
     */
    public function __construct(string $incomeDetails, User $deletedByUser)
    {
        $this->incomeDetails = $incomeDetails;
        $this->deletedByUser = $deletedByUser;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
                    ->subject('Other Income Record Deleted')
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("An other income record has been deleted from the system by **{$this->deletedByUser->name}**.")
                    ->line("Details of the deleted income: **{$this->incomeDetails}**")
                    ->line('Please review your income records.')
                    ->action('View All Other Incomes', url('/other-incomes'))
                    ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification for database storage.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'other_income_deleted',
            'income_details' => $this->incomeDetails,
            'deleted_by_user_id' => $this->deletedByUser->id,
            'deleted_by_user_name' => $this->deletedByUser->name,
            'message' => "Other income '{$this->incomeDetails}' deleted by {$this->deletedByUser->name}.",
            'link' => url('/other-incomes'),
        ];
    }
}