<?php

namespace App\Notifications;

use App\Models\User;
use App\Models\Expense;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class ExpenseRecordedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $expense;
    public $recordedByUser;

    /**
     * Create a new notification instance.
     */
    public function __construct(Expense $expense, User $recordedByUser)
    {
        $this->expense = $expense;
        $this->recordedByUser = $recordedByUser;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
                    ->subject('New Expense Recorded')
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("A new expense has been recorded in the system by **{$this->recordedByUser->name}** ({$this->recordedByUser->role}).")
                    ->line("Details:")
                    ->line("- **Category:** {$this->expense->category_name}")
                    ->line("- **Amount:** TZS " . number_format($this->expense->amount, 2))
                    ->line("- **Date:** {$this->expense->expense_date->format('Y-m-d')}")
                    ->lineIf($this->expense->other_details, "- **Other Details:** {$this->expense->other_details}")
                    ->action('View Expense', url('/expenses/' . $this->expense->id))
                    ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification for database storage.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        $message = "New expense of TZS " . number_format($this->expense->amount, 2) . " for '{$this->expense->category_name}' recorded by {$this->recordedByUser->name}.";
        if ($this->expense->other_details) {
            $message .= " (Details: {$this->expense->other_details})";
        }

        return [
            'type' => 'expense_recorded',
            'expense_id' => $this->expense->id,
            'category_name' => $this->expense->category_name,
            'amount' => $this->expense->amount,
            'recorded_by_user_id' => $this->recordedByUser->id,
            'recorded_by_user_name' => $this->recordedByUser->name,
            'message' => $message,
            'link' => url('/expenses/' . $this->expense->id),
        ];
    }
}