<?php

namespace App\Notifications;

use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class ExpenseDeletedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $expenseDetails; // String yenye maelezo ya expense iliyofutwa
    public $deletedByUser;

    /**
     * Create a new notification instance.
     */
    public function __construct(string $expenseDetails, User $deletedByUser)
    {
        $this->expenseDetails = $expenseDetails;
        $this->deletedByUser = $deletedByUser;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
                    ->subject('Expense Deleted')
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("An expense has been deleted from the system by **{$this->deletedByUser->name}** ({$this->deletedByUser->role}).")
                    ->line("Details of the deleted expense: **{$this->expenseDetails}**")
                    ->line('Please review your expense records.')
                    ->action('View All Expenses', url('/expenses'))
                    ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification for database storage.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'expense_deleted',
            'expense_details' => $this->expenseDetails,
            'deleted_by_user_id' => $this->deletedByUser->id,
            'deleted_by_user_name' => $this->deletedByUser->name,
            'message' => "Expense '{$this->expenseDetails}' deleted by {$this->deletedByUser->name}.",
            'link' => url('/expenses'),
        ];
    }
}