<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Auth\Passwords\CanResetPassword;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;

class User extends Authenticatable implements CanResetPasswordContract
{
    use HasApiTokens, HasFactory, Notifiable, CanResetPassword;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role', 
        'profile_photo_path',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
    ];

    public function products()
    {
        
        return $this->hasMany(Product::class, 'user_id'); // Au 'registered_by_user_id' kama ndilo jina la column
    }


    public function stockInputs()
    {
        return $this->hasMany(StockInput::class);
    }

    public function initiatedStockOutputs()
    {
        return $this->hasMany(StockOutput::class, 'output_by_user_id');
    }

    public function receivedStockOutputs()
    {
        return $this->hasMany(StockOutput::class, 'output_to_user_id');
    }

    public function sales()
    {
        return $this->hasMany(Sale::class, 'sold_by_user_id');
    }

    public function isAdmin()
    {
    return $this->role === 'admin';
    }
    public function isStorekeeper()
    {
        return $this->role === 'storekeeper';
    }

    public function isSeller()
    {
        return $this->role === 'seller';
    }

    public function isCleaner()
    {
        return $this->role === 'cleaner';
    }

    public function isStaff()
    {
        return $this->role === 'staff';
    }

        public function otherIncomes() 
    {
        return $this->hasMany(OtherIncome::class);
    }
   
    public function salaryPayments()
    {
        return $this->hasMany(SalaryPayment::class);
    }

    public function expenses()
    {
        return $this->hasMany(Expense::class);
    }

    public function loginHistories()
{
    return $this->hasMany(LoginHistory::class);
}


public function getProfilePhotoUrlAttribute()
    {
        return $this->profile_photo_path
                    ? asset('storage/' . $this->profile_photo_path)
                    : asset('images/default_profile.png'); // Default image if none exists
    }
}