<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class Product extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'quantity',
        'unit',
        'input_price_per_unit',
        'selling_price_per_unit',
        'user_id'
    ];

    public function owner()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function stockTransfers()
    {
        return $this->hasMany(StockTransfer::class);
    }

    public function stockOutputs()
    {
        return $this->hasMany(StockOutput::class);
    }

    public function sales()
    {
        return $this->hasMany(Sale::class);
    }

    public function stockInputs()
    {
        return $this->hasMany(StockInput::class);
    }
// app/Models/Product.php

public function getRemainingStockAttribute()
{
    return $this->quantity; // Or calculate based on inputs/outputs if needed
    // If you need to calculate based on stock inputs/outputs:
    // return $this->stockInputs()->sum('quantity') - $this->stockOutputs()->sum('quantity');
}

public function currentQuantity()
{
    return $this->quantity; // Or calculate based on inputs/outputs if needed
    // Alternative if you track inputs/outputs separately:
    // return $this->stockInputs()->sum('quantity') - $this->stockOutputs()->sum('quantity');
}

public function getAvailableQuantityForSellerAttribute()
{
    if (auth()->user()->role !== 'seller') {
        return $this->quantity;
    }
    
    // Pata stock zote zilizopokelewa kutoka kwa watumiaji wote
    $transferred = $this->stockOutputs()
        ->where('output_to_user_id', auth()->id())
        ->sum('quantity');
        
    // Pata mauzo yote yaliyofanywa na mtu huyu
    $sold = $this->sales()
        ->where('sold_by_user_id', auth()->id())
        ->sum('quantity');
        
    return $transferred - $sold;
}
// app/Models/Product.php
// ... (existing methods like hasMany for StockInput, StockOutput, Sale)

public function saleReturns() // Ensure this is still present
{
    return $this->hasMany(SaleReturn::class);
}

public function purchaseReturns() // Ensure this is still present
{
    return $this->hasMany(PurchaseReturn::class);
}




    /**
     * Calculate the total current inventory value for all products
     * as of a specific date, based on the cost price (input_price_per_unit).
     *
     * @param Carbon $asOfDate The date for which to calculate inventory.
     * @return float The total inventory value.
     */
   public static function getInventoryValue(Carbon $asOfDate): float
{
    $products = Product::whereHas('stockInputs', function($query) use ($asOfDate) {
        $query->where('created_at', '<=', $asOfDate);
    })->get();

    $totalValue = 0;

    foreach ($products as $product) {
        // Get all stock inputs before the date
        $inputs = $product->stockInputs()
            ->where('created_at', '<=', $asOfDate)
            ->get();

        // Get all stock outputs before the date
        $outputs = $product->stockOutputs()
            ->where('created_at', '<=', $asOfDate)
            ->get();

        // Calculate net quantity using FIFO method
        $remainingQuantity = 0;
        $remainingValue = 0;
        
        foreach ($inputs as $input) {
            $remainingQuantity += $input->quantity;
            $remainingValue += $input->quantity * $input->input_price_per_unit;
            
            // Deduct outputs
            foreach ($outputs as $output) {
                if ($output->quantity <= 0) continue;
                
                $deduct = min($input->quantity, $output->quantity);
                $remainingQuantity -= $deduct;
                $remainingValue -= $deduct * $input->input_price_per_unit;
                $output->quantity -= $deduct;
            }
        }

        $totalValue += $remainingValue;
    }

    return $totalValue;
}


    /**
     * Updates the current stock for a product based on inputs and outputs.
     * This method could be called after every stock input or output.
     */
    public function updateCurrentStock()
    {
        $totalInputs = $this->stockInputs()->sum('quantity');
        $totalOutputs = $this->stockOutputs()->sum('quantity');
        $totalSaleReturns = $this->saleReturns()->sum('quantity');
        $totalPurchaseReturns = $this->purchaseReturns()->sum('quantity');

        $this->current_stock = ($totalInputs - $totalOutputs + $totalSaleReturns - $totalPurchaseReturns);
        $this->save();
    }
    
}