<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class FixedAsset extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'purchase_date',
        'purchase_price',
        'useful_life_years', // Kwa depreciation
        'salvage_value',     // Kwa depreciation
        'user_id', // User aliyerekodi
    ];

    protected $casts = [
        'purchase_date' => 'datetime',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the annual depreciation amount for the asset (straight-line method).
     * @return float
     */
    public function getAnnualDepreciationAttribute(): float
    {
        if ($this->useful_life_years <= 0) {
            return 0.00;
        }
        return ($this->purchase_price - $this->salvage_value) / $this->useful_life_years;
    }

    /**
     * Get the accumulated depreciation as of the current date.
     * @return float
     */
    public function getAccumulatedDepreciationAttribute(): float
    {
        // If purchase date is in the future or no useful life
        if (!$this->purchase_date || $this->purchase_date->isFuture() || $this->useful_life_years <= 0) {
            return 0.00;
        }

        // Calculate years from purchase date to current date
        $yearsInUse = $this->purchase_date->diffInYears(Carbon::now()); // Use Carbon::now() for current accumulated depreciation

        // Ensure accumulated depreciation does not exceed depreciable amount (purchase_price - salvage_value)
        $maxDepreciation = $this->purchase_price - $this->salvage_value;
        $calculatedAccumulated = $this->annual_depreciation * $yearsInUse;

        return min($calculatedAccumulated, $maxDepreciation); // Cannot depreciate more than depreciable amount
    }

    /**
     * Get the net book value of the asset as of the current date.
     * @return float
     */
    public function getBookValueAttribute(): float
    {
        // Book value should not go below salvage value
        return max($this->salvage_value, $this->purchase_price - $this->accumulated_depreciation);
    }

    /**
     * Get the accumulated depreciation as of a *specific report date*.
     * This is crucial for Balance Sheet accuracy.
     *
     * @param \Carbon\Carbon $reportDate
     * @return float
     */
    public function getAccumulatedDepreciationAsOf(Carbon $reportDate): float
    {
        if (!$this->purchase_date || $this->purchase_date->greaterThan($reportDate) || $this->useful_life_years <= 0) {
            return 0.00; // Not purchased yet or no useful life
        }

        // Calculate years in use from purchase date up to the report date
        $yearsInUse = $this->purchase_date->diffInYears($reportDate);

        $maxDepreciation = $this->purchase_price - $this->salvage_value;
        $calculatedAccumulated = $this->annual_depreciation * $yearsInUse;

        return min($calculatedAccumulated, $maxDepreciation);
    }

    /**
     * Get the net book value of the asset as of a *specific report date*.
     * This is crucial for Balance Sheet accuracy.
     *
     * @param \Carbon\Carbon $reportDate
     * @return float
     */
    public function getBookValueAsOf(Carbon $reportDate): float
    {
        // Use the accumulated depreciation as of the report date
        $accumulatedDepreciation = $this->getAccumulatedDepreciationAsOf($reportDate);
        return max($this->salvage_value, $this->purchase_price - $accumulatedDepreciation);
    }
}
