<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rules\Enum;
use App\Enums\UserRole;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Str;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use App\Notifications\UserRegisteredNotification; 
use Illuminate\Support\Facades\Notification; 
use Illuminate\Support\Facades\Storage; 
use Carbon\Carbon;
use App\Models\Customer;
use App\Models\Sale;
use App\Models\Product;
use App\Models\StockInput;
use App\Models\StockOutput;
use App\Models\LoginHistory;

class UserController extends Controller
{
    /**
     * Display the admin dashboard with active users.
     * This method will serve as your admin dashboard route.
     *
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse
     */
  public function adminDashboard()
{
    if (Auth::user()->role !== 'admin') {
        abort(403, 'Unauthorized action.');
    }

    // Get active users
    $twoMinutesAgo = Carbon::now()->subMinutes(2)->timestamp;
    $activeSessionUserIds = DB::table('sessions')
        ->whereNotNull('user_id')
        ->where('last_activity', '>=', $twoMinutesAgo)
        ->pluck('user_id')
        ->unique();

    $currentlyLoggedInUsers = User::whereIn('id', $activeSessionUserIds)
                                ->orderBy('name')
                                ->get();

    // Get total customers
    $totalCustomers = Customer::count();

    // Get total sales data
    $totalSalesAmount = Sale::sum('total_price');
    $totalSalesCount = Sale::count();

    return view('admin.dashboard', [
        'currentlyLoggedInUsers' => $currentlyLoggedInUsers,
        'totalCustomers' => $totalCustomers,
        'totalSalesAmount' => $totalSalesAmount,
        'totalSalesCount' => $totalSalesCount
    ]);
}

    /**
     * Display the storekeeper dashboard with active users.
     *
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse
     */
    public function storekeeperDashboard(): View | RedirectResponse
    {
        if (Auth::user()->role !== 'storekeeper') {
            abort(403, 'Unauthorized action.');
        }

        // Define what "active" means (e.g., last activity within the last 2 minutes)
        $twoMinutesAgo = Carbon::now()->subMinutes(2)->timestamp;

        // Get user_ids from active sessions in the database
        $activeSessionUserIds = DB::table('sessions')
            ->whereNotNull('user_id')
            ->where('last_activity', '>=', $twoMinutesAgo)
            ->pluck('user_id')
            ->unique();

        // Fetch user details for these active user IDs
        $currentlyLoggedInUsers = User::whereIn('id', $activeSessionUserIds)
                                      ->orderBy('name')
                                      ->get();

        return view('storekeeper.dashboard', [
            'currentlyLoggedInUsers' => $currentlyLoggedInUsers,
        ]);
    }

    /**
     * Display the seller dashboard with active users.
     *
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse
     */
    public function sellerDashboard(): View | RedirectResponse
    {
        if (Auth::user()->role !== 'seller') {
            abort(403, 'Unauthorized action.');
        }

        // Define what "active" means (e.g., last activity within the last 2 minutes)
        $twoMinutesAgo = Carbon::now()->subMinutes(0.5)->timestamp;

        // Get user_ids from active sessions in the database
        $activeSessionUserIds = DB::table('sessions')
            ->whereNotNull('user_id')
            ->where('last_activity', '>=', $twoMinutesAgo)
            ->pluck('user_id')
            ->unique();

        // Fetch user details for these active user IDs
        $currentlyLoggedInUsers = User::whereIn('id', $activeSessionUserIds)
                                      ->orderBy('name')
                                      ->get();

        return view('seller.dashboard', [
            'currentlyLoggedInUsers' => $currentlyLoggedInUsers,
        ]);
    }

    /**
     * Display a listing of users.  (Admin only)
     *
     * @return \Illuminate\Http\Response
     */
   public function index(Request $request)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized action.');
        }

        $search = $request->input('search');
        
        $users = User::where('role', '!=', 'admin') // Exclude admin from listing
            ->when($search, function ($query) use ($search) {
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', '%'.$search.'%')
                      ->orWhere('email', 'like', '%'.$search.'%')
                      ->orWhere('role', 'like', '%'.$search.'%');
                });
            })
            ->orderBy('name')
            ->paginate(10);

        return view('admin.users.index', compact('users'));
    }
    /**
     * Show the form for creating a new user. (Admin only)
     *
     * @return \Illuminate\Http\Response
     */
    public function create(): View
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        
        $roles = [
            'storekeeper' => 'Storekeeper',
            'seller' => 'Seller',
            'cleaner' => 'Cleaner',
            'staff' => 'Staff'
        ];
        
        return view('auth.register', compact('roles'));
    }

    /**
     * Show the seller registration form. (Admin only)
     *
     * @return \Illuminate\Http\Response
     */
    public function showSellerRegistrationForm(): View
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        return view('auth.register');
    }

    /**
     * Show the storekeeper registration form. (Admin only)
     *
     * @return \Illuminate\Http\Response
     */
    public function showStorekeeperRegistrationForm(): View
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        return view('auth.register');
    }

    /**
     * Store a newly created user in storage. (Admin only)
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:storekeeper,seller,cleaner,staff',
            'profile_photo' => 'nullable|image|max:2048|mimes:jpeg,png,jpg,gif', 
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $profilePhotoPath = null;
        if ($request->hasFile('profile_photo')) {
            // Store the uploaded image in the 'public/profile-photos' directory
            $profilePhotoPath = $request->file('profile_photo')->store('profile-photos', 'public');
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            'email_verified_at' => now(),
            'remember_token' => Str::random(10),
            'profile_photo_path' => $profilePhotoPath,
        ]);

        event(new Registered($user));

       $adminUsers = User::where('role', 'admin')->get(); // Pata admins wote
$loggedInAdmin = Auth::user(); // Pata admin aliyesajili user

foreach ($adminUsers as $admin) {
    // Hakikisha admin hajitumi notification mwenyewe mara mbili kama ndiye aliyesajili
    // Unaweza kuamua kama admin aliyesajili apate noti moja tu kupitia dashboard
    // au apate email na dashboard. Kwa sasa tutatuma kwa wote.
    $admin->notify(new UserRegisteredNotification($user, $loggedInAdmin));
}


        return redirect()->route('users.index')->with('success', 'User registered successfully!');
    }
    /**
     * Display the specified user.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = User::findOrFail($id);
        return response()->json($user);
    }

    /**
     * Show the form for editing the specified user. (Admin Only)
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id): View
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }
        
        $user = User::findOrFail($id);
        $roles = [
            'storekeeper' => 'Storekeeper',
            'seller' => 'Seller',
            'cleaner' => 'Cleaner',
            'staff' => 'Staff'
        ];
        
        return view('admin.users.edit', compact('user', 'roles'));
    }
    /**
     * Update the specified user in storage. (Admin only)
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, User $user): RedirectResponse
    {
        if (!Auth::user()->isAdmin()) {
            abort(403, 'Unauthorized action.');
        }

        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id, // Unique email rule, ignoring current user's email
            'role' => 'required|in:storekeeper,seller,cleaner,staff',
            'profile_photo' => 'nullable|image|max:2048|mimes:jpeg,png,jpg,gif', // Validation for image
        ];

        // Only validate password if it's provided
        if ($request->filled('password')) {
            $rules['password'] = 'string|min:8|confirmed';
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = [
            'name' => $request->name,
            'email' => $request->email,
            'role' => $request->role,
        ];

        // Handle password update if provided
        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        // Handle profile photo upload
        if ($request->hasFile('profile_photo')) {
            // Futa picha ya zamani kama ipo
            if ($user->profile_photo_path) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }
            // Hifadhi picha mpya
            $data['profile_photo_path'] = $request->file('profile_photo')->store('profile-photos', 'public');
        } elseif ($request->boolean('remove_profile_photo')) { // Optional: A checkbox to explicitly remove photo
            if ($user->profile_photo_path) {
                Storage::disk('public')->delete($user->profile_photo_path);
                $data['profile_photo_path'] = null;
            }
        }


        $user->update($data);

        return redirect()->route('users.index')->with('success', 'User updated successfully!');
    }

    /**
     * Remove the specified user from storage. (Admin only)
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id): RedirectResponse
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized');
        }

        $user = User::findOrFail($id);
        $user->delete();

        return redirect()->route('users.index')->with('success', 'User deleted successfully!');
    }

    /**
     * Handle user login.  (For all roles)
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $credentials = $request->only('email', 'password');

        if (Auth::attempt($credentials)) {
            $user = Auth::user();
            $token = $user->createToken('auth_token')->plainTextToken;
            return response()->json([
                'message' => 'Logged in successfully',
                'user' => $user,
                'token' => $token,
            ], 200);
        }

        return response()->json(['message' => 'Invalid credentials'], 401);
    }

    /**
     * Handle user logout
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json(['message' => 'Logged out successfully'], 200);
    }

    /**
     * Display a listing of sellers. (Admin only)
     *
     * @return \Illuminate\Http\Response
     */
   public function showSellers(Request $request)
{
    if (Auth::user()->role !== 'admin') {
        abort(403, 'Unauthorized');
    }
    
    $search = $request->input('search');
    
    $users = User::where('role', 'seller')
        ->when($search, function ($query) use ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', '%'.$search.'%')
                  ->orWhere('email', 'like', '%'.$search.'%');
            });
        })
        ->orderBy('name')
        ->paginate(10);

    return view('admin.users.index', compact('users'));
}

public function showStorekeepers(Request $request)
{
    if (Auth::user()->role !== 'admin') {
        abort(403, 'Unauthorized');
    }
    
    $search = $request->input('search');
    
    $users = User::where('role', 'storekeeper')
        ->when($search, function ($query) use ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', '%'.$search.'%')
                  ->orWhere('email', 'like', '%'.$search.'%');
            });
        })
        ->orderBy('name')
        ->paginate(10);

    return view('admin.users.index', compact('users'));
}

 /**
     * Display the login history for the authenticated user.
     *
     * @return \Illuminate\View\View
     */
    public function showMyLoginHistory(): View
    {
        $user = Auth::user();
        $loginHistories = $user->loginHistories()->latest()->paginate(10); // Get latest 10 login records

        return view('profile.login-history', compact('loginHistories'));
    }

    /**
     * Display the login history for all users (Admin only).
     *
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse
     */
    public function showAllLoginHistory(): View | RedirectResponse
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized action.');
        }

        // Load login histories with associated user details
        $loginHistories = LoginHistory::with('user')
                                    ->latest('login_at') // Order by latest login first
                                    ->paginate(20); // Paginate for better performance

        return view('admin.login-history.index', compact('loginHistories'));
    }

public function showUserDetails($id)
{
    if (Auth::user()->role !== 'admin') {
        abort(403, 'Unauthorized action.');
    }

    $user = User::findOrFail($id);
    
    // Get user's products (stock they own)
    $products = Product::where('user_id', $user->id)->paginate(5, ['*'], 'products_page');
    
    // Get user's sales
    $sales = Sale::where('sold_by_user_id', $user->id)
                ->with(['product', 'customer'])
                ->latest()
                ->paginate(5, ['*'], 'sales_page');
    
    // Get stock inputs by this user
    $stockInputs = StockInput::where('user_id', $user->id)
                    ->with('product')
                    ->latest()
                    ->paginate(5, ['*'], 'inputs_page');
    
    // Get stock outputs (transfers) by this user
    $stockOutputs = StockOutput::where('output_by_user_id', $user->id)
                    ->with(['product', 'outputByUser', 'outputToUser'])
                    ->latest()
                    ->paginate(5, ['*'], 'outputs_page');
    
    // Get stock received by this user (transfers to them)
    $stockReceived = StockOutput::where('output_to_user_id', $user->id)
                    ->with(['product', 'outputByUser', 'outputToUser'])
                    ->latest()
                    ->paginate(5, ['*'], 'received_page');
    
    // Calculate total profit generated by this user
    $totalProfit = Sale::where('sold_by_user_id', $user->id)
                    ->sum(DB::raw('(selling_price_per_unit - (SELECT input_price_per_unit FROM products WHERE products.id = sales.product_id)) * quantity'));

    return view('admin.users.details', compact(
        'user', 
        'products',
        'sales',
        'stockInputs',
        'stockOutputs',
        'stockReceived',
        'totalProfit'
    ));
}

}