<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\StockOutput;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Notification; // MUHIMU
use App\Notifications\StockTransferredNotification; // Import
use App\Notifications\StockReceivedNotification;    // Import

class StockOutputController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'can:manage-stock-output']); // Only authorized users can access
    }

    /**
     * Display a listing of stock outputs.
     */
  public function index()
{
    $search = request('search');
    
    $query = StockOutput::with(['product', 'outputByUser', 'outputToUser']);
    
    if ($search) {
        $query->whereHas('product', function($q) use ($search) {
            $q->where('name', 'like', '%'.$search.'%');
        })->orWhereHas('outputToUser', function($q) use ($search) {
            $q->where('name', 'like', '%'.$search.'%');
        });
    }
    
    if (Auth::user()->role === 'storekeeper') {
        $query->where('output_by_user_id', Auth::id());
    } 
   
    elseif (Auth::user()->role === 'admin') {
        
    }
    
    elseif (Auth::user()->role === 'seller') {
        $query->where('output_to_user_id', Auth::id());
    }
    
    $stockOutputs = $query->latest()->paginate(10);
    
    return view('stock_outputs.index', compact('stockOutputs'));
}
    /**
     * Show the form for creating a new stock output.
     */
   public function create()
    {
        // Query products based on user role
        $productsQuery = Product::query();

        if (Auth::user()->role === 'storekeeper' || Auth::user()->role === 'admin') {
            // Storekeeper na Admin wanaona tu bidhaa walizozimiliki
            $productsQuery->where('user_id', Auth::id());
        } else {
            // Kwa roles zingine zisizoruhusiwa kutransfer, rudisha empty collection
            // Au unaweza ku-abort(403) hapa kama hawapaswi kuona fomu kabisa
            $productsQuery->whereRaw('1 = 0'); // Hakuna product zitakazorudishwa
        }

        $products = $productsQuery->get();
        
        // Filter users based on who can receive stock outputs
        $users = User::query();
        if (Auth::user()->role === 'admin') {
            $users->whereIn('role', ['seller', 'storekeeper']);
        } elseif (Auth::user()->role === 'storekeeper') {
            $users->where('role', 'seller'); // Storekeeper anaweza kutransfer kwa seller tu
        } else {
            // Roles zingine haziwezi kutransfer, rudisha empty collection
            $users->whereRaw('1 = 0');
        }
        $users = $users->get();

        return view('stock_outputs.create', compact('products', 'users'));
    }
    /**
     * Store a newly created stock output in storage.
     */
  public function store(Request $request)
{
    $request->validate([
        'product_id' => 'required|exists:products,id',
        'quantity' => 'required|numeric|min:0.01',
        'output_to_user_id' => 'required|exists:users,id',
        'reason' => 'nullable|string|max:255',
    ]);

     $product = Product::findOrFail($request->product_id);
    $quantityTransferred = $request->quantity;
    $toUser = User::findOrFail($request->output_to_user_id);

    $fromUser = Auth::user(); 

    
    if (!$fromUser) {
      
        return redirect()->back()->with('error', 'You must be logged in to perform this action.');
    }

    

    // Verify ownership for storekeeper
    if (Auth::user()->role === 'storekeeper' && $product->user_id != Auth::id()) {
        return back()->with('error', 'You can only transfer stock that you own.');
    }

    // Check sufficient stock
    if ($product->quantity < $request->quantity) {
        return back()->with('error', 'Insufficient stock available.');
    }

    // Deduct from sender's stock
    $product->quantity -= $request->quantity;
    $product->save();

    // Create stock output record
    $output = StockOutput::create([
        'product_id' => $product->id,
        'quantity' => $request->quantity,
        'output_by_user_id' => Auth::id(),
        'output_to_user_id' => $request->output_to_user_id,
        'reason' => $request->reason,
        'is_transfer' => true 
    ]);

    // Add to recipient's stock
    $recipientProduct = Product::firstOrCreate(
        ['name' => $product->name, 'user_id' => $request->output_to_user_id],
        [
            'unit' => $product->unit,
            'input_price_per_unit' => $product->input_price_per_unit,
            'selling_price_per_unit' => $product->selling_price_per_unit,
            'quantity' => 0
        ]
    );

    $recipientProduct->quantity += $request->quantity;
    $recipientProduct->save();

     // --- NOTIFICATIONS HAPA ---

    // 1. Anayetuma (Admin/Storekeeper) apate notification
    $fromUser->notify(new StockTransferredNotification($product, $quantityTransferred, $fromUser, $toUser));

    // 2. Anayepokea (Seller/Storekeeper) apate notification
    $toUser->notify(new StockReceivedNotification($recipientProduct, $quantityTransferred, $fromUser, $toUser));

    // 3. Ikiwa anayetuma ni Storekeeper, Admin pia apate notification (juu ya uhamishaji)
    if ($fromUser->isStorekeeper()) {
        $admins = User::where('role', 'admin')->get();
        Notification::send($admins, new StockTransferredNotification($product, $quantityTransferred, $fromUser, $toUser));
    }

    // --- MWISHO WA NOTIFICATIONS ---

    return redirect()->route('stock-outputs.index')->with('success', 'Stock transferred successfully.');
}
    /**
     * Show the form for editing the specified stock output.
     */
    public function edit(StockOutput $stockOutput)
    {
        $products = Product::all();
        $users = User::whereIn('role', ['seller', 'storekeeper'])->get();
        return view('stock_outputs.edit', compact('stockOutput', 'products', 'users'));
    }

    /**
     * Update the specified stock output in storage.
     */
    public function update(Request $request, StockOutput $stockOutput)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|numeric|min:0.01',
            'output_to_user_id' => 'nullable|exists:users,id',
            'reason' => 'nullable|string|max:255',
        ]);

        $originalQuantity = $stockOutput->quantity;
        $newQuantity = $request->quantity;
        $product = Product::findOrFail($request->product_id);

        // Calculate the difference and adjust product quantity
        $quantityDifference = $newQuantity - $originalQuantity;

        if ($product->quantity < $quantityDifference) {
            return back()->with('error', 'Cannot update: Not enough stock for the adjusted quantity.');
        }

        $product->quantity -= $quantityDifference;
        $product->save();

        $stockOutput->update([
            'product_id' => $request->product_id,
            'quantity' => $newQuantity,
            'output_to_user_id' => $request->output_to_user_id,
            'reason' => $request->reason,
        ]);

        return redirect()->route('stock-outputs.index')->with('success', 'Stock output updated successfully.');
    }

    /**
     * Remove the specified stock output from storage.
     */
    public function destroy(StockOutput $stockOutput)
    {
        // Revert stock quantity when deleting an output
        $product = $stockOutput->product;
        if ($product) {
            $product->quantity += $stockOutput->quantity;
            $product->save();
        }

        $stockOutput->delete();
        return redirect()->route('stock-outputs.index')->with('success', 'Stock output deleted successfully and quantity reverted.');
    }
}