<?php

namespace App\Http\Controllers;

use App\Models\StockTransfer;
use App\Models\StockOutput;
use App\Models\Sale;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SellerSalesController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'can:sell-stock']);
    }

    public function index()
    {
        $search = request('search');
        
        $query = Sale::with(['product', 'customer'])
            ->where('sold_by_user_id', Auth::id());
            
        if ($search) {
            $query->whereHas('product', function($q) use ($search) {
                $q->where('name', 'like', '%'.$search.'%');
            })->orWhereHas('customer', function($q) use ($search) {
                $q->where('username', 'like', '%'.$search.'%');
            });
        }
        
        $sales = $query->latest()->paginate(10);

        return view('sales.seller-index', compact('sales'));
    }

    public function availableStock()
    {
        $search = request('search');
        
        $query = Product::whereHas('stockOutputs', function($query) {
                $query->where('output_to_user_id', Auth::id());
            })
            ->with(['stockOutputs' => function($query) {
                $query->where('output_to_user_id', Auth::id())
                      ->selectRaw('product_id, SUM(quantity) as total_transferred')
                      ->groupBy('product_id');
            }]);
            
        if ($search) {
            $query->where('name', 'like', '%'.$search.'%');
        }
            
        $products = $query->get()
            ->map(function($product) {
                $product->available_quantity = $product->stockOutputs->sum('total_transferred') - 
                    $product->sales()->where('sold_by_user_id', Auth::id())->sum('quantity');
                return $product;
            })
            ->filter(function($product) {
                return $product->available_quantity > 0;
            });

        return view('sales.seller-stock', compact('products'));
    }

   public function receivedStock()
{
    $search = request('search');
    
    // Get both StockTransfers and StockOutputs where seller received stock
    $query = StockOutput::with(['product', 'outputByUser'])
              ->where('output_to_user_id', Auth::id());
    
    if ($search) {
        $query->whereHas('product', function($q) use ($search) {
            $q->where('name', 'like', '%'.$search.'%');
        })->orWhereHas('outputByUser', function($q) use ($search) {
            $q->where('name', 'like', '%'.$search.'%');
        });
    }
    
    $receivedStock = $query->latest()->paginate(10);
    
    return view('sales.received-stock', compact('receivedStock'));
}
}