<?php

namespace App\Http\Controllers;

use App\Models\SaleReturn;
use App\Models\Product;
use App\Models\Sale; // Kama unaunganisha na sale maalum
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class SaleReturnController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'can:sell-stock']); // Seller, Storekeeper au Admin
    }

    public function index()
    {
        $saleReturns = SaleReturn::with('product', 'sale', 'returnedBy')->latest()->paginate(10);
        return view('sale_returns.index', compact('saleReturns'));
    }

    public function create()
    {
        $products = Product::all();
        $sales = Sale::all(); // Kama utahitaji kuchagua sale ID
        return view('sale_returns.create', compact('products', 'sales'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
            'refund_amount' => 'required|numeric|min:0',
            'return_date' => 'required|date',
            'reason' => 'nullable|string',
            'sale_id' => 'nullable|exists:sales,id', // Validates if provided
        ]);

        DB::transaction(function () use ($request) {
            $product = Product::findOrFail($request->product_id);

            // Rekodi Sale Return
            SaleReturn::create([
                'sale_id' => $request->sale_id,
                'product_id' => $request->product_id,
                'quantity' => $request->quantity,
                'refund_amount' => $request->refund_amount,
                'return_date' => $request->return_date,
                'reason' => $request->reason,
                'returned_by_user_id' => Auth::id(),
            ]);

            // Ongeza stock tena kwenye Product
            $product->stock_on_hand += $request->quantity;
            $product->save();
        });

        return redirect()->route('sale-returns.index')->with('success', 'Sale Return recorded successfully.');
    }
    // show, edit, update, destroy methods kama utahitaji.
    public function show(SaleReturn $saleReturn)
    {
        return view('sale_returns.show', compact('saleReturn'));
    }

    public function edit(SaleReturn $saleReturn)
    {
        $products = Product::all();
        $sales = Sale::all();
        return view('sale_returns.edit', compact('saleReturn', 'products', 'sales'));
    }

    public function update(Request $request, SaleReturn $saleReturn)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
            'refund_amount' => 'required|numeric|min:0',
            'return_date' => 'required|date',
            'reason' => 'nullable|string',
            'sale_id' => 'nullable|exists:sales,id',
        ]);

        DB::transaction(function () use ($request, $saleReturn) {
            $oldQuantity = $saleReturn->quantity;
            $product = Product::findOrFail($request->product_id);

            $saleReturn->update([
                'sale_id' => $request->sale_id,
                'product_id' => $request->product_id,
                'quantity' => $request->quantity,
                'refund_amount' => $request->refund_amount,
                'return_date' => $request->return_date,
                'reason' => $request->reason,
            ]);

            // Rekebisha stock_on_hand ya bidhaa
            $product->stock_on_hand += ($request->quantity - $oldQuantity);
            $product->save();
        });

        return redirect()->route('sale-returns.index')->with('success', 'Sale Return updated successfully.');
    }

    public function destroy(SaleReturn $saleReturn)
    {
        DB::transaction(function () use ($saleReturn) {
            $product = $saleReturn->product;
            $product->stock_on_hand -= $saleReturn->quantity; // Punguza stock tena
            $product->save();
            $saleReturn->delete();
        });

        return redirect()->route('sale-returns.index')->with('success', 'Sale Return deleted successfully.');
    }
}