<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\Product;
use App\Models\User;
use App\Models\Sale;
use App\Models\StockOutput;
use App\Models\MonthlySaleReport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Notification;
use App\Notifications\SaleRecordedNotification;
use App\Notifications\SaleDeletedNotification;

class SaleController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'can:sell-stock']);
    }

    public function index()
    {
        $search = request('search');
        
        $query = Sale::with(['product', 'customer', 'soldBy']);
        
        if ($search) {
            $query->whereHas('product', function($q) use ($search) {
                $q->where('name', 'like', '%'.$search.'%');
            })->orWhereHas('customer', function($q) use ($search) {
                $q->where('username', 'like', '%'.$search.'%');
            });
        }
        
        // For seller, show only their sales
        if (Auth::user()->role === 'seller') {
            $query->where('sold_by_user_id', Auth::id());
        }
        // For storekeeper, show their sales and sales from their transfers
        elseif (Auth::user()->role === 'storekeeper') {
            $query->where(function($q) {
                $q->where('sold_by_user_id', Auth::id())
                  ->orWhereHas('product', function($q) {
                      $q->where('user_id', Auth::id());
                  });
            });
        }
        
        $sales = $query->latest()->paginate(10);
        
        return view('sales.index', compact('sales'));
    }

    public function create()
    {
        // ... code existing ...

        if (request()->has('product_id')) {
            $product = Product::findOrFail(request('product_id'));

            if (Auth::user()->role === 'seller') {
                // Calculate actual available quantity for the seller
                $transferredQuantity = StockOutput::where('product_id', $product->id)
                                                 ->where('output_to_user_id', Auth::id())
                                                 ->sum('quantity');

                $soldQuantity = Sale::where('product_id', $product->id)
                                     ->where('sold_by_user_id', Auth::id())
                                     ->sum('quantity');

                $availableQuantity = $transferredQuantity - $soldQuantity;

                if ($availableQuantity <= 0) {
                    return back()->with('error', 'You do not have sufficient stock of this product, or it is out of stock for you.');
                }

                // Create a "dummy" product object with the seller's available quantity
                // This ensures the form displays the correct quantity.
                $product->quantity = $availableQuantity; // Override the product quantity for this view
                $products = collect([$product]);

            } else {
                // For admin/storekeeper, check their own stock (existing logic)
                $query = Product::where('id', $product->id);

                if (Auth::user()->role === 'storekeeper') {
                    $query->where('user_id', Auth::id());
                }

                $product = $query->firstOrFail();

                if ($product->quantity <= 0) {
                    abort(403, 'This product is out of stock');
                }
                $products = collect([$product]);
            }
        } else {
            if (Auth::user()->role === 'seller') {
                // Seller sees only products they have received and still have available
                // This is similar to the logic in SellerSalesController@availableStock
                $products = Product::whereHas('stockOutputs', function($query) {
                                $query->where('output_to_user_id', Auth::id());
                            })
                            ->get()
                            ->map(function($product) {
                                $transferred = StockOutput::where('product_id', $product->id)
                                                            ->where('output_to_user_id', Auth::id())
                                                            ->sum('quantity');
                                $sold = Sale::where('product_id', $product->id)
                                             ->where('sold_by_user_id', Auth::id())
                                             ->sum('quantity');
                                $product->available_quantity = $transferred - $sold; // Set a new property for available quantity
                                return $product;
                            })
                            ->filter(function($product) {
                                return $product->available_quantity > 0;
                            });

                // Make sure the view receives the correct quantity
                $products->each(function($product) {
                    $product->quantity = $product->available_quantity;
                });

            } elseif (Auth::user()->role === 'storekeeper') {
                // Storekeeper sees only their own products
                $products = Product::where('user_id', Auth::id())
                    ->where('quantity', '>', 0)
                    ->get();
            } else {
                // Admin sees all products (with their main stock quantity)
                $products = Product::where('user_id', Auth::id()) // Ensure admin sees products they input
                    ->where('quantity', '>', 0)
                    ->get();
            }
        }

        return view('sales.create', compact('products'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|numeric|min:0.01',
            'customer_username' => 'required|string|max:255',
            'customer_phone_number' => 'nullable|string|max:20',
        ]);

        $product = Product::findOrFail($request->product_id);

        if (Auth::user()->role === 'seller') {
            $availableQuantity = $this->getSellerAvailableQuantity($product);
            
            if ($availableQuantity < $request->quantity) {
                return back()->with('error', 'You only have ' . $availableQuantity . ' ' . $product->unit . ' available of this product');
            }
        } else {
            // For admin/storekeeper, check if they own the product
            if (Auth::user()->role === 'storekeeper' && $product->user_id != Auth::id()) {
                return back()->with('error', 'You can only sell stock that you own.');
            }
            
            if ($product->quantity < $request->quantity) {
                return back()->with('error', 'Sale quantity exceeds current stock available for this product.');
            }
        }

        try {
            $customer = Customer::firstOrCreate(
                ['username' => $request->customer_username],
                ['phone_number' => $request->customer_phone_number]
            );

            $totalPrice = $request->quantity * $product->selling_price_per_unit;

            $sale = Sale::create([
                'product_id' => $product->id,
                'customer_id' => $customer->id,
                'sold_by_user_id' => Auth::id(),
                'quantity' => $request->quantity,
                'selling_price_per_unit' => $product->selling_price_per_unit,
                'total_price' => $totalPrice,
            ]);

            // Deduct from appropriate stock
            if (Auth::user()->role !== 'seller') {
                // For admin/storekeeper, deduct from their product quantity
                $product->quantity -= $request->quantity;
                $product->save();
            }

            // --- NOTIFICATIONS HAPA ---
            $loggedInUser = Auth::user();

            // 1. Mtu aliyeloga (admin, storekeeper, au seller) apate notification
            $loggedInUser->notify(new SaleRecordedNotification($sale, $loggedInUser));

            // 2. Admin na Storekeeper wote wapate notification (ikiwa si wao waliofanya sale)
            $adminsAndStorekeepers = User::whereIn('role', ['admin', 'storekeeper'])->get();
            // Tuma kwa admins na storekeepers wote, isipokuwa kama loggedInUser mwenyewe ni mmoja wao na tayari amepokea notification
            Notification::send($adminsAndStorekeepers->except($loggedInUser->id), new SaleRecordedNotification($sale, $loggedInUser));
            // --- MWISHO WA NOTIFICATIONS ---

            return redirect()->route('sales.index')->with('success', 'Sale recorded successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Error recording sale: ' . $e->getMessage());
        }
    }

    public function destroy(Sale $sale)
    {
        // Only allow deletion if user is admin or the one who made the sale
        if (Auth::user()->role !== 'admin' && $sale->sold_by_user_id !== Auth::id()) {
            abort(403, 'Unauthorized action.');
        }

        // Revert stock quantity when deleting a sale
        $product = $sale->product;
        if ($product) {
            $product->quantity += $sale->quantity;
            $product->save();
        }

        $sale->delete();

        // --- NOTIFICATIONS HAPA ---
        $deletedByUser = Auth::user();
        $saleDetails = $sale;

        // 1. Mtu aliyefuta (admin au muuzaji wa awali) apate notification
        $deletedByUser->notify(new SaleDeletedNotification($saleDetails, $deletedByUser));

        // 2. Admin na Storekeeper wote wapate notification (ikiwa si wao waliofuta)
        $adminsAndStorekeepers = User::whereIn('role', ['admin', 'storekeeper'])->get();
        // Tuma kwa admins na storekeepers wote, isipokuwa kama loggedInUser mwenyewe ni mmoja wao na tayari amepokea notification
        Notification::send($adminsAndStorekeepers->except($deletedByUser->id), new SaleDeletedNotification($saleDetails, $deletedByUser));
        // --- MWISHO WA NOTIFICATIONS ---

        return redirect()->route('sales.index')->with('success', 'Sale deleted successfully and quantity reverted.');
    }

    private function getSellerAvailableProducts()
    {
        $productIds = StockOutput::where('output_to_user_id', Auth::id())
            ->groupBy('product_id')
            ->havingRaw('SUM(quantity) > COALESCE((SELECT SUM(quantity) FROM sales WHERE product_id = stock_outputs.product_id AND sold_by_user_id = ?), 0)', [Auth::id()])
            ->pluck('product_id');
            
        return Product::whereIn('id', $productIds)->get();
    }

    private function getSellerAvailableQuantity($product)
    {
        // Calculate transferred quantity for this product to the current seller
        $transferredQuantity = StockOutput::where('product_id', $product->id)
                                         ->where('output_to_user_id', Auth::id())
                                         ->sum('quantity');

        // Calculate sold quantity for this product by the current seller
        $soldQuantity = Sale::where('product_id', $product->id)
                             ->where('sold_by_user_id', Auth::id())
                             ->sum('quantity');

        return $transferredQuantity - $soldQuantity;
    }

    public function totalSales(Request $request)
{
    if (!Gate::allows('is-admin')) {
        abort(403, 'Unauthorized action.');
    }

    $dateRange = $this->getDateRange($request);
    $startDate = $dateRange['startDate'];
    $endDate = $dateRange['endDate'];
    $periodLabel = $dateRange['periodLabel'];

    $totalSales = Sale::whereBetween('created_at', [$startDate, $endDate])
                     ->sum('total_price');
    
    $totalTransactions = Sale::whereBetween('created_at', [$startDate, $endDate])
                           ->count();

    $sales = Sale::with(['product', 'customer', 'soldBy'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->latest()
                ->paginate(10);

    return view('sales.total', compact(
        'totalSales',
        'totalTransactions',
        'periodLabel',
        'sales',
        'startDate',
        'endDate'
    ));
}

protected function getDateRange(Request $request): array
{
    $period = $request->input('period', 'month');
    
    switch ($period) {
        case 'today':
            $startDate = Carbon::today();
            $endDate = Carbon::today()->endOfDay();
            $periodLabel = 'Today: ' . $startDate->format('M d, Y');
            break;
        case 'week':
            $startDate = Carbon::now()->startOfWeek();
            $endDate = Carbon::now()->endOfWeek();
            $periodLabel = 'This Week: ' . $startDate->format('M d') . ' - ' . $endDate->format('M d');
            break;
        case 'year':
            $startDate = Carbon::now()->startOfYear();
            $endDate = Carbon::now()->endOfYear();
            $periodLabel = 'This Year: ' . $startDate->format('Y');
            break;
        case 'custom':
            $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
            $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
            $periodLabel = 'Custom Range: ' . $startDate->format('M d, Y') . ' - ' . $endDate->format('M d, Y');
            break;
        default: // month
            $startDate = Carbon::now()->startOfMonth();
            $endDate = Carbon::now()->endOfMonth();
            $periodLabel = 'This Month: ' . $startDate->format('M Y');
    }

    return [
        'startDate' => $startDate,
        'endDate' => $endDate,
        'periodLabel' => $periodLabel
    ];
}

    public function show($id)
    {
        if ($id === 'total') {
            return $this->totalSales();
        }

        $sale = Sale::findOrFail($id);
        return view('sales.show', compact('sale'));
    }

    public function salesReport(Request $request)
    {
        $period = $request->get('period', 'month');
        $title = 'Ripoti ya Mauzo';

        $now = Carbon::now();
        switch ($period) {
            case 'day':
                $startDate = Carbon::today();
                $endDate = Carbon::today()->endOfDay();
                $title = 'Ripoti ya Mauzo ya Leo';
                break;
            case 'week':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                $title = 'Ripoti ya Mauzo ya Wiki Hii';
                break;
            case 'month':
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                $title = 'Ripoti ya Mauzo ya Mwezi Huu';
                break;
            case 'quarter':
                $startDate = Carbon::now()->startOfQuarter();
                $endDate = Carbon::now()->endOfQuarter();
                $title = 'Ripoti ya Mauzo ya Robo Mwaka';
                break;
            case 'half_year':
                $startDate = Carbon::now()->month <= 6 
                    ? Carbon::now()->startOfYear() 
                    : Carbon::now()->startOfYear()->addMonths(6);
                $endDate = Carbon::now()->month <= 6 
                    ? Carbon::now()->startOfYear()->addMonths(5)->endOfMonth() 
                    : Carbon::now()->endOfYear();
                $title = 'Ripoti ya Mauzo ya Nusu Mwaka';
                break;
            case 'year':
            default:
                $startDate = Carbon::now()->startOfYear();
                $endDate = Carbon::now()->endOfYear();
                $title = 'Ripoti ya Mauzo ya Mwaka';
                break;
        }

        $sales = Sale::with(['customer', 'product'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->orderBy('created_at', 'desc')
            ->get();

        $totalSales = $sales->sum('total_price');
        $totalProfit = $sales->sum(function ($sale) {
            return ($sale->selling_price_per_unit - $sale->product->input_price_per_unit) * $sale->quantity;
        });

        if ($request->ajax()) {
            return response()->json([
                'html' => view('sales.report_partial', compact('sales', 'title', 'totalSales', 'totalProfit'))->render(),
                'totalSales' => number_format($totalSales),
                'totalProfit' => number_format($totalProfit)
            ]);
        }

        return view('sales.report', compact('sales', 'title', 'totalSales', 'totalProfit', 'period'));
    }

    public function printReport(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date'
        ]);

        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();

        $title = 'Sales Report ' . $startDate->format('d/m/Y') . ' - ' . $endDate->format('d/m/Y');

        $sales = Sale::with(['customer', 'product'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->orderBy('created_at', 'desc')
            ->get();

        $totalSales = $sales->sum('total_price');
        $totalProfit = $sales->sum(function ($sale) {
            return ($sale->selling_price_per_unit - $sale->product->input_price_per_unit) * $sale->quantity;
        });

        return view('sales.print_report', [
            'sales' => $sales,
            'title' => $title,
            'totalSales' => $totalSales,
            'totalProfit' => $totalProfit,
            'startDate' => $startDate->format('d/m/Y'),
            'endDate' => $endDate->format('d/m/Y')
        ]);
    }

    public function closeMonthlySales(Request $request)
    {
        $request->validate([
            'month' => 'required|numeric|between:1,12',
            'year' => 'required|numeric'
        ]);

        $month = $request->month;
        $year = $request->year;

        $startDate = Carbon::create($year, $month, 1)->startOfMonth();
        $endDate = Carbon::create($year, $month, 1)->endOfMonth();

        $existingReport = MonthlySaleReport::where('month', $month)
            ->where('year', $year)
            ->exists();

        if ($existingReport) {
            return response()->json([
                'success' => false,
                'message' => 'Ripoti ya mwezi huu tayari imefungwa'
            ]);
        }

        $sales = Sale::whereBetween('created_at', [$startDate, $endDate])->get();

        $totalSales = $sales->sum('total_price');
        $totalProfit = $sales->sum(function ($sale) {
            return ($sale->selling_price_per_unit - $sale->product->input_price_per_unit) * $sale->quantity;
        });

        $report = MonthlySaleReport::create([
            'month' => $month,
            'year' => $year,
            'total_sales' => $totalSales,
            'total_profit' => $totalProfit,
            'closed_by' => auth()->id(),
            'closed_at' => now()
        ]);

        $pdf = PDF::loadView('sales.monthly_report', [
            'sales' => $sales,
            'report' => $report,
            'monthName' => $startDate->monthName,
            'year' => $year
        ]);

        $filename = 'monthly_report_'.$month.'_'.$year.'_'.now()->format('YmdHis').'.pdf';
        Storage::put('reports/'.$filename, $pdf->output());

        $report->update(['report_path' => 'reports/'.$filename]);

        return response()->json([
            'success' => true,
            'message' => 'Mauzo ya mwezi yamefungwa kikamilifu',
            'download_url' => route('sales.download-monthly-report', $report->id)
        ]);
    }

    public function downloadMonthlyReport($id)
    {
        $report = MonthlySaleReport::findOrFail($id);
        return Storage::download($report->report_path);
    }

    public function salesReportWithCalendar(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->endOfMonth()->format('Y-m-d'));

        $title = 'Ripoti ya Mauzo ' . Carbon::parse($startDate)->format('d/m/Y') . ' - ' . Carbon::parse($endDate)->format('d/m/Y');

        $sales = Sale::with(['customer', 'product'])
            ->whereDate('created_at', '>=', $startDate)
            ->whereDate('created_at', '<=', $endDate)
            ->orderBy('created_at', 'desc')
            ->get();

        $totalSales = $sales->sum('total_price');
        $totalProfit = $sales->sum(function ($sale) {
            if ($sale->product) {
                return ($sale->selling_price_per_unit - $sale->product->input_price_per_unit) * $sale->quantity;
            }
            return 0;
        });

        return view('sales.report_with_calendar', compact('sales', 'title', 'totalSales', 'totalProfit', 'startDate', 'endDate'));
    }

    public function downloadPdfReport(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date'
        ]);

        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();

        $title = 'Sales Report ' . $startDate->format('d/m/Y') . ' - ' . $endDate->format('d/m/Y');

        $sales = Sale::with(['customer', 'product'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->orderBy('created_at', 'desc')
            ->get();

        $totalSales = $sales->sum('total_price');
        $totalProfit = $sales->sum(function ($sale) {
            return ($sale->selling_price_per_unit - $sale->product->input_price_per_unit) * $sale->quantity;
        });

        $pdf = PDF::loadView('sales.print_report', [
            'sales' => $sales,
            'title' => $title,
            'totalSales' => $totalSales,
            'totalProfit' => $totalProfit,
            'startDate' => $startDate->format('d/m/Y'),
            'endDate' => $endDate->format('d/m/Y')
        ]);

        return $pdf->download('sales_report_'.$startDate->format('Ymd').'_'.$endDate->format('Ymd').'.pdf');
    }
}