<?php

namespace App\Http\Controllers;

use App\Models\SalaryPayment;
use App\Models\User; // Tunahitaji kuleta model ya User ili kuchagua watumiaji
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification; // MUHIMU
use App\Notifications\SalaryPaidNotification;
use Illuminate\Support\Facades\Gate;

class SalaryController extends Controller
{
    public function __construct()
    {
        // Hakikisha ni Admin pekee ndiye anayeweza kufikia methods hizi
        $this->middleware(['auth', 'can:is-admin']); // Tutafafanua middleware hii baadaye
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $salaryPayments = SalaryPayment::with('user')
                                        ->latest('payment_date')
                                        ->paginate(10);
        return view('salaries.index', compact('salaryPayments'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $users = User::all(); // Pata watumiaji wote wanaoweza kulipwa
        return view('salaries.create', compact('users'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'payment_period' => 'nullable|string|max:255',
        ]);

        SalaryPayment::create($request->all());

        // --- NOTIFICATIONS HAPA ---
    $loggedInAdmin = Auth::user();

    // Tuma notification kwa admin wote
    $admins = User::where('role', 'admin')->get();
    Notification::send($admins, new SalaryPaidNotification($salaryPayment, $loggedInAdmin));
    // --- MWISHO WA NOTIFICATIONS ---


        return redirect()->route('salaries.index')->with('success', 'Malipo ya mshahara yameongezwa kikamilifu.');
    }

    /**
     * Display the specified resource.
     */
    public function show(SalaryPayment $salary)
    {
        return view('salaries.show', compact('salary'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(SalaryPayment $salary)
    {
        $users = User::all(); // Pata watumiaji wote
        return view('salaries.edit', compact('salary', 'users'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, SalaryPayment $salary)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'payment_period' => 'nullable|string|max:255',
        ]);

        $salary->update($request->all());

        return redirect()->route('salaries.index')->with('success', 'Malipo ya mshahara yamebadilishwa kikamilifu.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(SalaryPayment $salary)
    {
        $salary->delete();

        return redirect()->route('salaries.index')->with('success', 'Malipo ya mshahara yamefutwa kikamilifu.');
    }
}