<?php

namespace App\Http\Controllers;

use App\Models\{
    StockInput,
    StockOutput,
    Product,
    Sale,
    Expense,
    OtherIncome,
    SalaryPayment,
    User,
    SaleReturn,
    PurchaseReturn
};
use Illuminate\Http\Request;
use Illuminate\Support\Facades\{Auth, Gate, DB};
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;

class ReportController extends Controller
{
    public function __construct()
    {
        // Hakikisha ni Admin pekee ndiye anayeweza kufikia ripoti hizi
        $this->middleware(['auth', 'can:is-admin']);
    }

    /**
     * Huonyesha dashboard ya ripoti.
     * @return \Illuminate\Contracts\View\View
     */
    public function reportsDashboard()
    {
        $months = [];
        for ($i = 1; $i <= 12; $i++) {
            $months[$i] = Carbon::create()->month($i)->format('F');
        }

        $years = range(now()->year, now()->year - 5); // Miaka 5 iliyopita kutoka sasa
        return view('reports.dashboard', compact('months', 'years'));
    }

    /**
     * Huzalisha ripoti ya Stock Inputs kwa kipindi maalum.
     * @param Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function stockInputsReport(Request $request)
    {
        $dateRange = $this->getDateRange($request);

        $inputs = StockInput::with(['product', 'user'])
            ->whereBetween('created_at', [$dateRange['startDate'], $dateRange['endDate']])
            ->orderBy('created_at', 'desc')
            ->get();

        $totals = [
            'quantity' => $inputs->sum('quantity'),
            'value' => $inputs->sum(fn($input) => $input->quantity * $input->input_price_per_unit)
        ];

        return view('reports.stock_inputs', [
            'inputs' => $inputs,
            'totals' => $totals,
            'startDate' => $dateRange['startDate']->format('Y-m-d'),
            'endDate' => $dateRange['endDate']->format('Y-m-d'),
            'periodLabel' => $dateRange['periodLabel']
        ]);
    }

    /**
     * Huzalisha ripoti ya Stock Outputs kwa kipindi maalum.
     * @param Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function stockOutputsReport(Request $request)
    {
        $dateRange = $this->getDateRange($request);

        $outputs = StockOutput::with(['product', 'outputByUser', 'outputToUser'])
            ->whereBetween('created_at', [$dateRange['startDate'], $dateRange['endDate']])
            ->orderBy('created_at', 'desc')
            ->get();

        $totals = [
            'quantity' => $outputs->sum('quantity'),
            'value' => $outputs->sum(fn($output) => $output->quantity * $output->input_price_per_unit)
        ];

        return view('reports.stock_outputs', [
            'outputs' => $outputs,
            'totals' => $totals,
            'startDate' => $dateRange['startDate']->format('Y-m-d'),
            'endDate' => $dateRange['endDate']->format('Y-m-d'),
            'periodLabel' => $dateRange['periodLabel']
        ]);
    }

    /**
     * Huzalisha ripoti ya mauzo yote kwa kipindi maalum.
     * @param Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function totalSalesReport(Request $request)
    {
        $dateRange = $this->getDateRange($request);

        $sales = Sale::with(['customer', 'product', 'soldBy'])
            ->whereBetween('created_at', [$dateRange['startDate'], $dateRange['endDate']])
            ->orderBy('created_at', 'desc')
            ->paginate(25);

        $totals = [
            'sales' => Sale::whereBetween('created_at', [$dateRange['startDate'], $dateRange['endDate']])->sum('total_price'),
            'transactions' => Sale::whereBetween('created_at', [$dateRange['startDate'], $dateRange['endDate']])->count()
        ];

        return view('sales.total', [
            'sales' => $sales,
            'totalSales' => $totals['sales'],
            'totalTransactions' => $totals['transactions'],
            'startDate' => $dateRange['startDate']->format('Y-m-d'),
            'endDate' => $dateRange['endDate']->format('Y-m-d'),
            'periodLabel' => $dateRange['periodLabel']
        ]);
    }

    /**
     * Hupata tarehe za kuanzia na kumalizia kulingana na ombi.
     * @param Request $request
     * @return array
     */
    protected function getDateRange(Request $request): array
    {
        $period = $request->input('period', 'month');

        switch ($period) {
            case 'today':
                $startDate = Carbon::today();
                $endDate = Carbon::today()->endOfDay();
                $periodLabel = 'Today: ' . $startDate->format('M d, Y');
                break;
            case 'week':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                $periodLabel = 'This Week: ' . $startDate->format('M d') . ' - ' . $endDate->format('M d');
                break;
            case 'year':
                $startDate = Carbon::now()->startOfYear();
                $endDate = Carbon::now()->endOfYear();
                $periodLabel = 'This Year: ' . $startDate->format('Y');
                break;
            case 'custom':
                $request->validate([
                    'start_date' => 'required|date',
                    'end_date' => 'required|date|after_or_equal:start_date',
                ]);
                $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
                $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
                $periodLabel = 'Custom Range: ' . $startDate->format('M d, Y') . ' - ' . $endDate->format('M d, Y');
                break;
            default: // month
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                $periodLabel = 'This Month: ' . $startDate->format('M Y');
        }

        return [
            'startDate' => $startDate,
            'endDate' => $endDate,
            'periodLabel' => $periodLabel
        ];
    }

    /**
     * Huzalisha Taarifa ya Mapato (Income Statement).
     * @param Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function generateIncomeStatement(Request $request)
    {
        $reportData = $this->calculateIncomeStatementData($request);
        return view('reports.income_statement', [
            'reportData' => $reportData,
            'startDate' => $reportData['startDate'],
            'endDate' => $reportData['endDate'],
            'periodLabel' => $reportData['periodLabel']
        ]);
    }

    /**
     * Huhesabu data kwa ajili ya Taarifa ya Mapato.
     * @param Request $request
     * @return array
     */
    protected function calculateIncomeStatementData(Request $request): array
    {
        $dateRange = $this->getDateRange($request);
        $startDate = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];
        $periodLabel = $dateRange['periodLabel'];

        // Mapato
        $grossSales = Sale::whereBetween('created_at', [$startDate, $endDate])->sum('total_price');
        $salesReturns = SaleReturn::whereBetween('return_date', [$startDate, $endDate])->sum('refund_amount');
        $otherIncomes = OtherIncome::whereBetween('income_date', [$startDate, $endDate])->sum('amount');
        $netSales = $grossSales - $salesReturns + $otherIncomes;

        // Gharama ya Bidhaa Zilizouzwa (Cost of Goods Sold - COGS)
        // Inventory ya Mwanzo kwa kipindi hiki
        $openingInventory = Product::getInventoryValue($startDate->copy()->subDay());
        // Inventory ya Mwisho kwa kipindi hiki
        $closingInventory = Product::getInventoryValue($endDate);

        $totalPurchasesValue = StockInput::whereBetween('created_at', [$startDate, $endDate])
            ->sum(DB::raw('quantity * input_price_per_unit'));

        $purchaseReturnsCost = PurchaseReturn::whereBetween('return_date', [$startDate, $endDate])
            ->sum(DB::raw('quantity * (SELECT COALESCE(input_price_per_unit, 0) FROM products WHERE products.id = purchase_returns.product_id)'));

        $carriageInwards = Expense::whereBetween('expense_date', [$startDate, $endDate])
            ->where('category_name', 'Carriage Inwards')
            ->sum('amount');

        $cogs = $openingInventory + ($totalPurchasesValue - $purchaseReturnsCost + $carriageInwards) - $closingInventory;
        $cogs = max(0, $cogs); // Hakikisha COGS haina thamani hasi

        $grossProfit = $netSales - $cogs;

        // Gharama za Uendeshaji (Operating Expenses)
        // Panga gharama kwa kategoria kwa ajili ya kuonyesha kwa undani
        $operatingExpensesGrouped = Expense::whereBetween('expense_date', [$startDate, $endDate])
            ->whereNotIn('category_name', ['Carriage Inwards']) // Carriage Inwards tayari iko kwenye COGS
            ->groupBy('category_name')
            ->select('category_name', DB::raw('SUM(amount) as total_amount'))
            ->pluck('total_amount', 'category_name');

        $totalGeneralOperatingExpenses = $operatingExpensesGrouped->sum(); // Jumla ya gharama zote zilizopangwa

        $salaries = SalaryPayment::whereBetween('payment_date', [$startDate, $endDate])->sum('amount');

        // Carriage Outwards ni gharama ya uendeshaji
        $carriageOutwards = Expense::whereBetween('expense_date', [$startDate, $endDate])
            ->where('category_name', 'Carriage Outwards')
            ->sum('amount');

        $totalOperatingExpenses = $totalGeneralOperatingExpenses + $salaries + $carriageOutwards;
        $netProfit = $grossProfit - $totalOperatingExpenses;

        // Ongeza Carriage Outwards kwenye gharama zilizopangwa kwa ajili ya kuonyesha ikiwa ipo
        if ($carriageOutwards > 0) {
            $operatingExpensesGrouped->put('Carriage Outwards', $carriageOutwards);
        }

        return [
            'title' => 'Income Statement',
            'grossSales' => $grossSales,
            'salesReturns' => $salesReturns,
            'otherIncomes' => $otherIncomes,
            'netSales' => $netSales,
            'openingInventory' => $openingInventory,
            'totalPurchasesValue' => $totalPurchasesValue,
            'purchaseReturnsCost' => $purchaseReturnsCost,
            'carriageInwards' => $carriageInwards,
            'closingInventory' => $closingInventory,
            'cogs' => $cogs,
            'grossProfit' => $grossProfit,
            'operatingExpenses' => $operatingExpensesGrouped,
            'salaries' => $salaries,
            'carriageOutwards' => $carriageOutwards,
            'totalOperatingExpenses' => $totalOperatingExpenses,
            'netProfit' => $netProfit,
            'periodLabel' => $periodLabel,
            'startDate' => $startDate->format('Y-m-d'),
            'endDate' => $endDate->format('Y-m-d'),
        ];
    }

    /**
     * Huzalisha Taarifa ya Hali ya Kifedha (Balance Sheet).
     * @param Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function generateBalanceSheet(Request $request)
    {
        $reportDate = $request->input('report_date')
            ? Carbon::parse($request->input('report_date'))->endOfDay()
            : Carbon::now()->endOfDay();

        $reportData = $this->calculateBalanceSheetData($reportDate);
        return view('reports.balance_sheet', compact('reportData'));
    }

    /**
     * Huhesabu data kwa ajili ya Taarifa ya Hali ya Kifedha (Balance Sheet).
     * @param Carbon $reportDate
     * @return array
     */
    protected function calculateBalanceSheetData(Carbon $reportDate): array
    {
        // Assets
        $inventory = Product::getInventoryValue($reportDate);
        $cash = $this->calculateCashBalance($reportDate);
        $totalCurrentAssets = $inventory + $cash;
        $totalNonCurrentAssets = 0; // Kudhani hakuna fixed assets kwa mfumo huu
        $totalAssets = $totalCurrentAssets + $totalNonCurrentAssets;

        // Liabilities (Kudhani mfumo wa cash-based bila akaunti za kulipa/mikopo)
        $totalCurrentLiabilities = 0;
        $totalNonCurrentLiabilities = 0;
        $totalLiabilities = $totalCurrentLiabilities + $totalNonCurrentLiabilities;

        // Equity
        // Mtaji wa Mwanzo (Initial Capital) - Umejumuishwa kama 0 kwa sasa.
        // Katika mfumo halisi, inapaswa kuwa mchango halisi wa mtaji.
        $initialCapital = 0.00;

        // Faida Iliyohifadhiwa (Retained Earnings) ni jumla ya faida halisi kutoka mwanzo
        // wa biashara hadi tarehe ya ripoti.
        $retainedEarnings = $this->calculateCumulativeNetProfit($reportDate);

        // Faida ya Mwezi wa Sasa (kwa ajili ya kuonyesha tu, haiongezi kwenye totalEquity mara mbili)
        $currentMonthStartDate = $reportDate->copy()->startOfMonth();
        $currentMonthEndDate = $reportDate->copy()->endOfMonth();
        $tempRequestForCurrentMonthProfit = new Request([
            'period' => 'custom',
            'start_date' => $currentMonthStartDate->format('Y-m-d'),
            'end_date' => $currentMonthEndDate->format('Y-m-d'),
        ]);
        $currentMonthIncomeStatement = $this->calculateIncomeStatementData($tempRequestForCurrentMonthProfit);
        $currentMonthProfit = $currentMonthIncomeStatement['netProfit'];

        // Jumla ya Equity (Mtaji + Faida Iliyohifadhiwa)
        $totalEquity = $initialCapital + $retainedEarnings;

        // Kuangalia Usawa: Assets = Liabilities + Equity
        $balanceCheck = $totalAssets - ($totalLiabilities + $totalEquity);
        $isBalanced = abs($balanceCheck) < 0.01; // Ruhusu tofauti ndogo za namba za desimali

        return [
            'title' => 'Statement of Financial Position',
            'reportDate' => $reportDate->format('F j, Y'),
            'assets' => [
                'current' => [
                    'inventory' => $inventory,
                    'cash' => $cash,
                    'total' => $totalCurrentAssets
                ],
                'nonCurrent' => [
                    'fixedAssets' => 0,
                    'total' => $totalNonCurrentAssets
                ],
                'totalAssets' => $totalAssets
            ],
            'liabilities' => [
                'current' => [
                    'accountsPayable' => 0,
                    'total' => $totalCurrentLiabilities
                ],
                'nonCurrent' => [
                    'loans' => 0,
                    'total' => $totalNonCurrentLiabilities
                ],
                'totalLiabilities' => $totalLiabilities
            ],
            'equity' => [
                'capital' => $initialCapital,
                'retainedEarnings' => $retainedEarnings,
                'currentProfit' => $currentMonthProfit,
                'totalEquity' => $totalEquity
            ],
            'balanceCheck' => $balanceCheck,
            'isBalanced' => $isBalanced
        ];
    }

    /**
     * Huhesabu faida halisi iliyohifadhiwa (cumulative net profit) kutoka mwanzo wa biashara
     * hadi tarehe maalum.
     *
     * @param Carbon $endDate Tarehe ya mwisho ya kuhesabu.
     * @return float Faida halisi iliyohifadhiwa.
     */
    protected function calculateCumulativeNetProfit(Carbon $endDate): float
    {
        $businessStartDate = $this->getEarliestTransactionDate();

        if (!$businessStartDate || $businessStartDate->greaterThan($endDate)) {
            return 0.00; // Hakuna data au kipindi kisicho sahihi.
        }

        // Unda Request object ya muda ili kupata data ya mapato kwa kipindi chote cha biashara
        $tempRequest = new Request([
            'period' => 'custom',
            'start_date' => $businessStartDate->format('Y-m-d'),
            'end_date' => $endDate->format('Y-m-d'),
        ]);

        $cumulativeIncomeStatement = $this->calculateIncomeStatementData($tempRequest);

        return $cumulativeIncomeStatement['netProfit'];
    }

    /**
     *
     * @param Carbon $date Tarehe ambayo salio la pesa taslimu linahesabiwa.
     * @return float Salio la pesa taslimu.
     */
    protected function calculateCashBalance(Carbon $date): float
    {
        $businessStart = $this->getEarliestTransactionDate();

        if (!$businessStart) {
            return 0.00; // Hakuna miamala, hivyo salio la pesa taslimu ni 0
        }

        // Jumla ya mapato ya pesa taslimu kutoka mwanzo wa biashara hadi tarehe maalum
        $salesCash = Sale::whereBetween('created_at', [$businessStart, $date])->sum('total_price');
        $otherIncomeCash = OtherIncome::whereBetween('income_date', [$businessStart, $date])->sum('amount');
        $purchaseReturnsCash = PurchaseReturn::whereBetween('return_date', [$businessStart, $date])->sum('refund_received_amount');

        // Jumla ya matumizi ya pesa taslimu kutoka mwanzo wa biashara hadi tarehe maalum
        $stockPurchasesCash = StockInput::whereBetween('created_at', [$businessStart, $date])
            ->sum(DB::raw('quantity * input_price_per_unit'));
        $expensesCash = Expense::whereBetween('expense_date', [$businessStart, $date])->sum('amount');
        $salaryPaymentsCash = SalaryPayment::whereBetween('payment_date', [$businessStart, $date])->sum('amount');
        $saleReturnsRefundedCash = SaleReturn::whereBetween('return_date', [$businessStart, $date])->sum('refund_amount');

        $totalInflows = $salesCash + $otherIncomeCash + $purchaseReturnsCash;
        $totalOutflows = $stockPurchasesCash + $expensesCash + $salaryPaymentsCash + $saleReturnsRefundedCash;

        return $totalInflows - $totalOutflows;
    }

    /**
     * Huzalisha ripoti kamili ya kifedha ya kila mwezi.
     * @param Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function monthlyFinancialReport(Request $request)
    {
        $request->validate([
            'month' => 'required|numeric|between:1,12',
            'year' => 'required|numeric'
        ]);

        $month = $request->month;
        $year = $request->year;
        $startDate = Carbon::create($year, $month, 1)->startOfMonth();
        $endDate = Carbon::create($year, $month, 1)->endOfMonth();

        $tempRequest = new Request([
            'period' => 'custom',
            'start_date' => $startDate->format('Y-m-d'),
            'end_date' => $endDate->format('Y-m-d'),
        ]);

        $data = [
            'month' => $month,
            'year' => $year,
            'monthName' => $startDate->monthName,
            'startDate' => $startDate,
            'endDate' => $endDate,
            'incomeStatement' => $this->calculateIncomeStatementData($tempRequest),
            'balanceSheet' => $this->calculateBalanceSheetData($endDate),
            'stockReport' => $this->calculateStockReportData($startDate, $endDate)
        ];

        return view('reports.monthly-comprehensive', $data);
    }

    /**
     * Huzalisha Taarifa ya Mtiririko wa Pesa (Cash Flow Statement).
     * @param Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function generateCashFlowStatement(Request $request)
    {
        $dateRange = $this->getDateRange($request);
        $startDate = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];
        $periodLabel = $dateRange['periodLabel'];

        $cashFlowData = $this->calculateCashFlowData($startDate, $endDate);

        return view('reports.cash_flow_statement', [
            'reportData' => $cashFlowData,
            'startDate' => $startDate->format('Y-m-d'),
            'endDate' => $endDate->format('Y-m-d'),
            'periodLabel' => $periodLabel
        ]);
    }


    /**
     * Huhesabu data kwa ajili ya Taarifa ya Mtiririko wa Pesa.
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @return array
     */
    protected function calculateCashFlowData(Carbon $startDate, Carbon $endDate): array
    {
        $cashReceivedFromSales = Sale::whereBetween('created_at', [$startDate, $endDate])->sum('total_price');
        $cashReceivedOtherIncome = OtherIncome::whereBetween('income_date', [$startDate, $endDate])->sum('amount');
        $cashReceivedFromPurchaseReturns = PurchaseReturn::whereBetween('return_date', [$startDate, $endDate])->sum('refund_received_amount');

        $cashPaidForStockPurchases = StockInput::whereBetween('created_at', [$startDate, $endDate])
            ->sum(DB::raw('quantity * input_price_per_unit'));
        $expensesCash = Expense::whereBetween('expense_date', [$startDate, $endDate])->sum('amount');
        $salaryPaymentsCash = SalaryPayment::whereBetween('payment_date', [$startDate, $endDate])->sum('amount');
        $cashPaidForSaleRefunds = SaleReturn::whereBetween('return_date', [$startDate, $endDate])->sum('refund_amount');

        $totalInflows = $cashReceivedFromSales + $cashReceivedOtherIncome + $cashReceivedFromPurchaseReturns;
        $totalOutflows = $cashPaidForStockPurchases + $expensesCash + $salaryPaymentsCash + $cashPaidForSaleRefunds;

        $netCashFromOperatingActivities = ($totalInflows) - ($totalOutflows);

        $cashFromInvestingActivities = 0.00; 

        $cashFromFinancingActivities = 0.00; 

        $netIncreaseDecreaseInCash = $netCashFromOperatingActivities + $cashFromInvestingActivities + $cashFromFinancingActivities;

        $beginningCashBalance = $this->calculateCashBalance($startDate->copy()->subDay());

        $endingCashBalance = $beginningCashBalance + $netIncreaseDecreaseInCash;

        $verifiedEndingCashBalance = $this->calculateCashBalance($endDate);

        $isCashFlowBalanced = abs($endingCashBalance - $verifiedEndingCashBalance) < 0.01;

        return [
            'title' => 'Cash Flow Statement',
            'operatingActivities' => [
                'cashReceivedFromSales' => $cashReceivedFromSales,
                'cashReceivedOtherIncome' => $cashReceivedOtherIncome,
                'cashReceivedFromPurchaseReturns' => $cashReceivedFromPurchaseReturns,
                'cashPaidForStockPurchases' => $cashPaidForStockPurchases,
                'cashPaidForExpenses' => $expensesCash,
                'cashPaidForSalaries' => $salaryPaymentsCash,
                'cashPaidForSaleRefunds' => $cashPaidForSaleRefunds,
                'netCash' => $netCashFromOperatingActivities,
            ],
            'investingActivities' => [
                'netCash' => $cashFromInvestingActivities,
            ],
            'financingActivities' => [
                'netCash' => $cashFromFinancingActivities,
            ],
            'netIncreaseDecreaseInCash' => $netIncreaseDecreaseInCash,
            'beginningCashBalance' => $beginningCashBalance,
            'endingCashBalance' => $endingCashBalance,
            'reconciliationCheck' => $verifiedEndingCashBalance,
            'isCashFlowBalanced' => $isCashFlowBalanced,
        ];
    }

    /**
     * Hupakua Taarifa ya Mtiririko wa Pesa kama PDF.
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function downloadCashFlowStatementPdf(Request $request)
    {
        $dateRange = $this->getDateRange($request);
        $startDate = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];
        $periodLabel = $dateRange['periodLabel'];

        $reportData = $this->calculateCashFlowData($startDate, $endDate);

        $pdf = PDF::loadView('reports.cash_flow_statement_pdf', [
            'reportData' => $reportData,
            'periodLabel' => $periodLabel
        ]);

        return $pdf->download('cash-flow-statement-' .
            $startDate->format('Ymd') . '-' .
            $endDate->format('Ymd') . '.pdf');
    }

    /**
     * Hupakua ripoti kamili ya kila mwezi kama PDF.
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function downloadMonthlyReportPdf(Request $request)
    {
        $request->validate([
            'month' => 'required|numeric|between:1,12',
            'year' => 'required|numeric'
        ]);

        $view = $this->monthlyFinancialReport($request);
        $data = $view->getData();

        $pdf = PDF::loadView('reports.pdf.monthly-comprehensive', (array) $data)
            ->setPaper('a4', 'portrait');

        return $pdf->download("financial-report-{$data['monthName']}-{$data['year']}.pdf");
    }

    /**
     * Hupakua Taarifa ya Mapato kama PDF.
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function downloadIncomeStatementPdf(Request $request)
    {
        $reportData = $this->calculateIncomeStatementData($request);
        $periodLabel = $reportData['periodLabel'];

        $pdf = PDF::loadView('reports.income_statement_pdf', [
            'reportData' => $reportData,
            'periodLabel' => $periodLabel
        ]);

        return $pdf->download('income-statement-' .
            Carbon::parse($reportData['startDate'])->format('Ymd') . '-' .
            Carbon::parse($reportData['endDate'])->format('Ymd') . '.pdf');
    }

    /**
     * Hupakua Taarifa ya Hali ya Kifedha kama PDF.
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function downloadBalanceSheetPdf(Request $request)
    {
        $reportDate = Carbon::parse($request->input('report_date', now()->format('Y-m-d')))->endOfDay();
        $reportData = $this->calculateBalanceSheetData($reportDate);

        $pdf = PDF::loadView('reports.balance_sheet_pdf', [
            'reportData' => $reportData,
            'periodLabel' => 'As of ' . $reportDate->format('M d, Y')
        ]);

        return $pdf->download('balance_sheet_' . now()->format('Ymd') . '.pdf');
    }

    /**
     * Huhesabu data ya ripoti ya stock.
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @return array
     */
    protected function calculateStockReportData(Carbon $startDate, Carbon $endDate): array
    {
        $inputs = StockInput::with('product')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->get();

        $outputs = StockOutput::with('product')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->get();

        $totalInputValue = $inputs->sum(fn($input) => $input->quantity * $input->input_price_per_unit);
        $totalOutputValue = $outputs->sum(fn($output) => $output->quantity * $output->input_price_per_unit);

        $openingStock = Product::getInventoryValue($startDate->copy()->subDay());
        $closingStock = Product::getInventoryValue($endDate);

        return [
            'inputs' => $inputs,
            'outputs' => $outputs,
            'totalInputQuantity' => $inputs->sum('quantity'),
            'totalInputValue' => $totalInputValue,
            'totalOutputQuantity' => $outputs->sum('quantity'),
            'totalOutputValue' => $totalOutputValue,
            'openingStock' => $openingStock,
            'closingStock' => $closingStock,
            'period' => $startDate->format('F Y')
        ];
    }

    /**
     * Huzalisha data kwa ajili ya Trial Balance hadi tarehe maalum.
     *
     * @param Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function generateTrialBalance(Request $request)
    {
        $reportDate = $request->input('report_date')
            ? Carbon::parse($request->input('report_date'))->endOfDay()
            : Carbon::now()->endOfDay();

        $trialBalanceData = $this->calculateTrialBalanceData($reportDate);

        return view('reports.trial_balance', [
            'trialBalanceData' => $trialBalanceData,
            'reportDate' => $reportDate->format('Y-m-d'), // Hii inaweza kutumika kama 'value' ya input
            'periodLabel' => 'As of ' . $reportDate->format('M d, Y') // Hii ni kwa ajili ya display tu
        ]);
    }

    /**
     * Huhesabu data ya trial balance hadi tarehe maalum ya ripoti.
     * Akaunti zote za mapato na matumizi zitaakisi kiasi jumla kutoka
     * mwanzo wa biashara hadi tarehe ya ripoti.
     *
     * @param Carbon $reportDate Tarehe ambayo trial balance inazalishwa.
     * @return array
     */
    protected function calculateTrialBalanceData(Carbon $reportDate): array
    {
        $earliestTransactionDate = $this->getEarliestTransactionDate();
        if (!$earliestTransactionDate) {
            // Hakuna miamala katika mfumo, rudisha trial balance tupu
            return $this->getEmptyTrialBalanceData($reportDate);
        }

        // Unda Request object ya muda ili kupata data ya income statement jumla
        // kutoka mwanzo wa biashara hadi tarehe ya ripoti.
        $tempRequest = new Request([
            'period' => 'custom',
            'start_date' => $earliestTransactionDate->format('Y-m-d'),
            'end_date' => $reportDate->format('Y-m-d'),
        ]);

        $cumulativeIncomeStatement = $this->calculateIncomeStatementData($tempRequest);

        // --- Salio za Debit ---
        $debits = [];

        // Assets
        $cashBalance = $this->calculateCashBalance($reportDate);
        // Ikiwa cashBalance ni hasi, basi ni overdraft na itaenda kwenye Credit, sio Debit.
        if ($cashBalance > 0) {
            $debits['Cash'] = $cashBalance;
        } else {
            $debits['Cash'] = 0.00;
        }
        $debits['Inventory'] = Product::getInventoryValue($reportDate); // Thamani ya cost ya inventory
        $debits['Sales Returns'] = $cumulativeIncomeStatement['salesReturns']; // Contra-revenue (huongeza Debits)

        // Expenses (Jumla kutoka mwanzo wa biashara hadi tarehe ya ripoti)
        $debits['Cost of Goods Sold'] = $cumulativeIncomeStatement['cogs'];
        $debits['Salaries Expense'] = $cumulativeIncomeStatement['salaries'];
        $debits['Carriage Outwards Expense'] = $cumulativeIncomeStatement['carriageOutwards'];

        // Ongeza gharama zingine za uendeshaji kutoka income statement (ukiondoa Carriage Inwards/Outwards)
        foreach ($cumulativeIncomeStatement['operatingExpenses'] as $category => $amount) {
            if (!in_array($category, ['Carriage Inwards', 'Carriage Outwards'])) {
                $debits[ucwords(str_replace('_', ' ', $category)) . ' Expense'] = $amount;
            }
        }


        // --- Salio za Credit ---
        $credits = [];

        // Liabilities
        // Ikiwa cashBalance ni hasi, basi ni overdraft na inawekwa kama liability (Credit).
        if ($cashBalance < 0) {
            $credits['Bank Overdraft'] = abs($cashBalance);
        } else {
            $credits['Bank Overdraft'] = 0.00;
        }
        $credits['Accounts Payable'] = 0.00; // Ikiwa unafuatilia akaunti za kulipa, ziweke hapa
        $credits['Loans Payable'] = 0.00;    // Ikiwa unafuatilia mikopo, iweke hapa

        $credits['Initial Capital'] = 0.00; 

        
        $credits['Sales Revenue'] = $cumulativeIncomeStatement['grossSales'];
        $credits['Other Income'] = $cumulativeIncomeStatement['otherIncomes'];
        $credits['Purchase Returns'] = $cumulativeIncomeStatement['purchaseReturnsCost']; // Contra-purchase (huongeza Credits)


        $debits = array_filter($debits, fn($value) => $value > 0);
        $credits = array_filter($credits, fn($value) => $value > 0);


      
        $totalDebits = array_sum($debits);
        $totalCredits = array_sum($credits);

     
        $isBalanced = abs($totalDebits - $totalCredits) < 0.01; // Ruhusu tofauti ndogo za namba za desimali

        return [
            'title' => 'Trial Balance',
            'reportDate' => $reportDate->format('F j, Y'),
            'accounts' => [
                'debits' => $debits,
                'credits' => $credits,
            ],
            'totalDebits' => $totalDebits,
            'totalCredits' => $totalCredits,
            'isBalanced' => $isBalanced,
            'balanceDifference' => $totalDebits - $totalCredits,
        ];
    }

    /**
     * Helper kupata tarehe ya mwanzo kabisa ya muamala kwenye models zote muhimu.
     *
     * @return Carbon|null
     */
    protected function getEarliestTransactionDate(): ?Carbon
    {
        $earliestStockInputDate = StockInput::min('created_at');
        $earliestSaleDate = Sale::min('created_at');
        $earliestExpenseDate = Expense::min('expense_date');
        $earliestOtherIncomeDate = OtherIncome::min('income_date');
        $earliestSalaryPaymentDate = SalaryPayment::min('payment_date');
        $earliestSaleReturnDate = SaleReturn::min('return_date');
        $earliestPurchaseReturnDate = PurchaseReturn::min('return_date');

        $allDates = array_filter([
            $earliestStockInputDate,
            $earliestSaleDate,
            $earliestExpenseDate,
            $earliestOtherIncomeDate,
            $earliestSalaryPaymentDate,
            $earliestSaleReturnDate,
            $earliestPurchaseReturnDate
        ]);

        if (empty($allDates)) {
            return null; // Hakuna data katika mfumo
        }

        return Carbon::parse(min($allDates))->startOfDay();
    }

    /**
     * Hurudisha muundo wa trial balance tupu ikiwa hakuna miamala iliyopo.
     *
     * @param Carbon $reportDate
     * @return array
     */
    protected function getEmptyTrialBalanceData(Carbon $reportDate): array
    {
        return [
            'title' => 'Trial Balance',
            'reportDate' => $reportDate->format('F j, Y'),
            'accounts' => [
                'debits' => [],
                'credits' => [],
            ],
            'totalDebits' => 0.00,
            'totalCredits' => 0.00,
            'isBalanced' => true,
            'balanceDifference' => 0.00,
        ];
    }

    /**
     * Hupakua Trial Balance kama PDF.
     * SASA HUONYESHA PDF KWENYE KIVINJARI KWANZA BADALA YA KUPAKUA MOJA KWA MOJA.
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function downloadTrialBalancePdf(Request $request)
    {
        $reportDate = Carbon::parse($request->input('report_date', now()->format('Y-m-d')))->endOfDay();
        $trialBalanceData = $this->calculateTrialBalanceData($reportDate);

        $pdf = PDF::loadView('reports.trial_balance_pdf', [
            'trialBalanceData' => $trialBalanceData,
            'periodLabel' => 'As of ' . $reportDate->format('M d, Y')
        ]);

        // Badilisha kutoka download() kwenda stream() ili ionyeshwe kwenye kivinjari kwanza
        return $pdf->stream('trial_balance_' . $reportDate->format('Ymd') . '.pdf');
    }

    // ReportController.php

// Kwa Income Statement
public function viewIncomeStatementPdf(Request $request)
{
    $reportData = $this->calculateIncomeStatementData($request);
    $periodLabel = $reportData['periodLabel'];
    
    return view('reports.income_statement_pdf', [
        'title' => 'Income Statement',
        'reportData' => $reportData,
        'periodLabel' => $periodLabel,
        'startDate' => $reportData['startDate'],
        'endDate' => $reportData['endDate']
    ]);
}

// Kwa Balance Sheet
public function viewBalanceSheetPdf(Request $request)
{
    $reportDate = Carbon::parse($request->input('report_date', now()->format('Y-m-d')))->endOfDay();
    $reportData = $this->calculateBalanceSheetData($reportDate);
    
    return view('reports.balance_sheet_pdf', [
        'title' => 'Balance Sheet',
        'reportData' => $reportData,
        'periodLabel' => 'As of ' . $reportDate->format('M d, Y'),
        'reportDate' => $reportDate->format('Y-m-d')
    ]);
}

// Kwa Cash Flow Statement
public function viewCashFlowStatementPdf(Request $request)
{
    $dateRange = $this->getDateRange($request);
    $startDate = $dateRange['startDate'];
    $endDate = $dateRange['endDate'];
    $periodLabel = $dateRange['periodLabel'];
    $cashFlowData = $this->calculateCashFlowData($startDate, $endDate);

    return view('reports.cash_flow_statement_pdf', [
        'title' => 'Cash Flow Statement',
        'reportData' => $cashFlowData,
        'periodLabel' => $periodLabel,
        'startDate' => $startDate->format('Y-m-d'),
        'endDate' => $endDate->format('Y-m-d')
    ]);
}

// Kwa Trial Balance
public function viewTrialBalancePdf(Request $request)
{
    $reportDate = Carbon::parse($request->input('report_date', now()->format('Y-m-d')))->endOfDay();
    $trialBalanceData = $this->calculateTrialBalanceData($reportDate);

    return view('reports.trial_balance_pdf', [
        'title' => 'Trial Balance',
        'trialBalanceData' => $trialBalanceData,
        'periodLabel' => 'As of ' . $reportDate->format('M d, Y'),
        'reportDate' => $reportDate->format('Y-m-d')
    ]);
}
}
