<?php

namespace App\Http\Controllers;

use App\Models\PurchaseReturn;
use App\Models\Product; // Tutahitaji Product ili kupata input_price_per_unit
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class PurchaseReturnController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'can:manage-stock']); // Storekeeper au Admin
    }

    public function index()
    {
        $purchaseReturns = PurchaseReturn::with('product', 'returnedBy')->latest()->paginate(10);
        return view('purchase_returns.index', compact('purchaseReturns'));
    }

    public function create()
    {
        $products = Product::all();
        return view('purchase_returns.create', compact('products'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
            'return_date' => 'required|date',
            'reason' => 'nullable|string',
            'refund_received_amount' => 'required|numeric|min:0',
        ]);

        DB::transaction(function () use ($request) {
            $product = Product::findOrFail($request->product_id);

            // Rekodi Purchase Return
            PurchaseReturn::create([
                'product_id' => $request->product_id,
                'quantity' => $request->quantity,
                'refund_received_amount' => $request->refund_received_amount,
                'return_date' => $request->return_date,
                'reason' => $request->reason,
                'returned_by_user_id' => Auth::id(),
            ]);

            // Ongeza stock tena kwenye Product (kama iliondolewa kimakosa au kwa uharibifu)
            // Au punguza StockInput iliyotangulia, kutegemea na logic ya biashara yako.
            // Kwa urahisi, tunarudisha stock kwenye Product.
            $product->stock_on_hand += $request->quantity; // Tunarudisha kwenye stock halisi.
            $product->save();
        });

        return redirect()->route('purchase-returns.index')->with('success', 'Purchase Return recorded successfully.');
    }

    public function show(PurchaseReturn $purchaseReturn)
    {
        return view('purchase_returns.show', compact('purchaseReturn'));
    }

    // Edit na Update methods kama utahitaji kuzibadilisha baadaye
    public function edit(PurchaseReturn $purchaseReturn)
    {
        $products = Product::all();
        return view('purchase_returns.edit', compact('purchaseReturn', 'products'));
    }

    public function update(Request $request, PurchaseReturn $purchaseReturn)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
            'return_date' => 'required|date',
            'reason' => 'nullable|string',
            'refund_received_amount' => 'required|numeric|min:0',
        ]);

        DB::transaction(function () use ($request, $purchaseReturn) {
            $oldQuantity = $purchaseReturn->quantity;
            $product = Product::findOrFail($request->product_id);

            $purchaseReturn->update([
                'product_id' => $request->product_id,
                'quantity' => $request->quantity,
                'refund_received_amount' => $request->refund_received_amount,
                'return_date' => $request->return_date,
                'reason' => $request->reason,
                // 'returned_by_user_id' => Auth::id(), // Don't change who returned unless necessary
            ]);

            // Rekebisha stock_on_hand ya bidhaa
            $product->stock_on_hand += ($request->quantity - $oldQuantity);
            $product->save();
        });

        return redirect()->route('purchase-returns.index')->with('success', 'Purchase Return updated successfully.');
    }

    public function destroy(PurchaseReturn $purchaseReturn)
    {
        DB::transaction(function () use ($purchaseReturn) {
            $product = $purchaseReturn->product;
            $product->stock_on_hand -= $purchaseReturn->quantity; // Punguza stock tena
            $product->save();
            $purchaseReturn->delete();
        });

        return redirect()->route('purchase-returns.index')->with('success', 'Purchase Return deleted successfully.');
    }
}