<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\user;
use App\Models\Sale;
use App\Models\StockInput;
use App\Models\StockTransfer;
use App\Models\StockOutput;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification; // MUHIMU: Hii ni kwa ajili ya kutuma notifications
use App\Notifications\StockAddedNotification; // Import notification classes ulizounda
use App\Notifications\StockDeletedNotification;
use App\Notifications\StockTransferredNotification;
use App\Notifications\StockReceivedNotification;

class ProductController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'can:manage-stock']); // Only authorized users can access
    }

    /**
     * Display a listing of the products (stock).
     */
public function index()
    {
        $search = request('search');
        
        $query = Product::query();

        // Filter products based on user role
        if (Auth::user()->role === 'seller') {
            // Seller anaona tu bidhaa walizopokea na bado anazo
            $query->where('user_id', Auth::id());
        } elseif (Auth::user()->role === 'storekeeper') {
            // Storekeeper anaona tu bidhaa alizozimiliki
            $query->where('user_id', Auth::id());
        } elseif (Auth::user()->role === 'admin') {
            // Admin anaona bidhaa zake zote za asili au alizoinject
            // Kama admin anapaswa kuona stock zote kwenye mfumo (hata za wengine),
            // basi hapa hakuna filter ya user_id.
            // LAKINI, kwa kuwa products zimegawanyika kwa user_id,
            // admin anapaswa kuona zile anazomiliki yeye mwenyewe kama "stock yake".
            $query->where('user_id', Auth::id()); 
        }

        if ($search) {
            $query->where('name', 'like', '%'.$search.'%');
        }

        $products = $query->latest()->paginate(10); // Au paginate unavyotaka

        return view('products.index', compact('products'));
    }
    /**
     * Show the form for creating a new product.
     */
    public function create()
    {
        return view('products.create');
    }

    /**
     * Store a newly created product in storage.
     */
   public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'quantity' => 'required|numeric|min:0',
            'unit' => 'required|string|in:kg,g,pieces,packages,amount,boxes',
            'input_price_per_unit' => 'required|numeric|min:0',
            'selling_price_per_unit' => 'required|numeric|min:0',
        ]);

        // Create the product
        $product = Product::create([
            'name' => $request->name,
            'quantity' => $request->quantity,
            'unit' => $request->unit,
            'input_price_per_unit' => $request->input_price_per_unit,
            'selling_price_per_unit' => $request->selling_price_per_unit,
            'user_id' => Auth::id(),
        ]);

        // Record the stock input
        StockInput::create([
            'product_id' => $product->id,
            'quantity' => $request->quantity,
            'input_price_per_unit' => $request->input_price_per_unit,
            'user_id' => Auth::id(),
        ]);

    // Send notifications
    $loggedInUser = Auth::user();
    $loggedInUser->notify(new StockAddedNotification($product, $request->quantity, $loggedInUser, true));

    if ($loggedInUser->isAdmin()) {
        $storekeepers = User::where('role', 'storekeeper')->get();
        Notification::send($storekeepers, new StockAddedNotification($product, $request->quantity, $loggedInUser, true));
    }

        return redirect()->route('products.index')->with('success', 'Product and stock added successfully.');
    }

    /**
     * Show the form for editing the specified product.
     */
    public function edit(Product $product)
    {
        return view('products.edit', compact('product'));
    }

    /**
     * Update the specified product in storage.
     */
    public function update(Request $request, Product $product)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:products,name,' . $product->id,
            'unit' => 'required|string|in:kg,g,pieces,packages,amount,boxes',
            'input_price_per_unit' => 'required|numeric|min:0',
            'selling_price_per_unit' => 'required|numeric|min:0',
        ]);

        // Note: Quantity update is usually handled by stock inputs/outputs, not directly here.
        // If you need to manually adjust current stock, consider a separate process or admin-only override.
        $product->update([
            'name' => $request->name,
            'unit' => $request->unit,
            'input_price_per_unit' => $request->input_price_per_unit,
            'selling_price_per_unit' => $request->selling_price_per_unit,
        ]);

        return redirect()->route('products.index')->with('success', 'Product updated successfully.');
    }

    /**
     * Remove the specified product from storage.
     */
    public function destroy(Product $product)
    {
        $product->delete();

          // --- NOTIFICATIONS HAPA ---
    // 1. Admin apate notification
    if ($deletedByUser->isAdmin()) {
        $allAdmins = User::where('role', 'admin')->get();
        Notification::send($allAdmins, new StockDeletedNotification($productName, $deletedByUser));
    }

    // 2. Storekeepers pia wapate notification kama ilifutwa na admin au storekeeper
    if ($deletedByUser->isAdmin() || $deletedByUser->isStorekeeper()) {
        $storekeepers = User::where('role', 'storekeeper')->get();
        // Epuka kutuma mara mbili kwa storekeeper aliyefanya kama yupo kwenye list
        $storekeepers = $storekeepers->reject(function ($user) use ($deletedByUser) {
            return $user->id === $deletedByUser->id;
        });
        Notification::send($storekeepers, new StockDeletedNotification($productName, $deletedByUser));
    }
    // --- MWISHO WA NOTIFICATIONS ---

        return redirect()->route('products.index')->with('success', 'Product deleted successfully.');
    }
public function showAddStockForm(Product $product)
{
    // Admin anaweza kuongeza stock kwa bidhaa yoyote
    if (Auth::user()->role === 'admin') {
        return view('products.add-stock', compact('product'));
    }
    
    // Storekeeper anaweza kuongeza stock kwenye bidhaa zake tu
    if (Auth::user()->role === 'storekeeper' && $product->user_id == Auth::id()) {
        return view('products.add-stock', compact('product'));
    }
    
    // Seller hawezi kuongeza stock
    abort(403, 'Unauthorized action.');
}

    /**
     * Add stock to a product.
     */
    public function addStock(Request $request, Product $product)
    {
        // Verify authorization
        if (Auth::user()->role === 'storekeeper' && $product->user_id != Auth::id()) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'quantity' => 'required|numeric|min:0.01',
            'input_price_per_unit' => 'required|numeric|min:0',
        ]);

        // Update product quantity and price
        $product->quantity += $request->quantity;
        $product->input_price_per_unit = $request->input_price_per_unit;
        $product->save();

        // Record the stock input
        StockInput::create([
            'product_id' => $product->id,
            'quantity' => $request->quantity,
            'input_price_per_unit' => $request->input_price_per_unit,
            'user_id' => Auth::id(),
        ]);

        // --- NOTIFICATIONS HAPA ---
    $loggedInUser = Auth::user();

    // 1. Mtu aliyeloga na kuongeza stock apate notification
    $loggedInUser->notify(new StockAddedNotification($product, $request->quantity, $loggedInUser, false)); // false kwa isNewProduct

    // 2. Ikiwa admin ndiye aliyefanya, storekeepers pia wapate notification
    if ($loggedInUser->isAdmin()) {
        $storekeepers = User::where('role', 'storekeeper')->get();
        // Epuka kutuma mara mbili kwa storekeeper aliyefanya kama yupo kwenye list ya storekeepers
        $storekeepers = $storekeepers->reject(function ($user) use ($loggedInUser) {
            return $user->id === $loggedInUser->id;
        });
        Notification::send($storekeepers, new StockAddedNotification($product, $request->quantity, $loggedInUser, false));
    }
    // --- MWISHO WA NOTIFICATIONS ---


        return redirect()->route('products.index')
            ->with('success', 'Stock added successfully.');
    }

 public function transferStock(Request $request)
{
    $request->validate([
        'product_id' => 'required|exists:products,id',
        'quantity' => 'required|numeric|min:0.01',
        'output_to_user_id' => 'required|exists:users,id',
        'reason' => 'nullable|string|max:255',
    ]);

    $product = Product::findOrFail($request->product_id);

    // Verify ownership for storekeeper
    if (Auth::user()->role === 'storekeeper' && $product->user_id != Auth::id()) {
        return back()->with('error', 'You can only transfer stock that you own.');
    }

    if ($product->quantity < $request->quantity) {
        return back()->with('error', 'Insufficient stock available.');
    }

    // Deduct from sender's stock
    $product->quantity -= $request->quantity;
    $product->save();

    // Find or create product for recipient (WITHOUT creating duplicate)
    $recipientProduct = Product::where('name', $product->name)
        ->where('user_id', $request->output_to_user_id)
        ->first();

    if (!$recipientProduct) {
        $recipientProduct = Product::create([
            'name' => $product->name,
            'unit' => $product->unit,
            'input_price_per_unit' => $product->input_price_per_unit,
            'selling_price_per_unit' => $product->selling_price_per_unit,
            'quantity' => 0,
            'user_id' => $request->output_to_user_id,
        ]);
    }

    $recipientProduct->quantity += $request->quantity;
    $recipientProduct->save();

    // Create transfer records
    StockTransfer::create([
        'product_id' => $product->id,
        'quantity' => $request->quantity,
        'from_user_id' => Auth::id(),
        'to_user_id' => $request->output_to_user_id,
    ]);

    StockOutput::create([
        'product_id' => $product->id,
        'quantity' => $request->quantity,
        'output_by_user_id' => Auth::id(),
        'output_to_user_id' => $request->output_to_user_id,
        'reason' => $request->reason,
        'is_transfer' => true,
    ]);

     // --- NOTIFICATIONS HAPA ---
    // 1. Anayetuma (Admin/Storekeeper) apate notification
    $fromUser->notify(new StockTransferredNotification($product, $quantityTransferred, $fromUser, $toUser));

    // 2. Anayepokea (Seller/Storekeeper) apate notification
    $toUser->notify(new StockReceivedNotification($recipientProduct, $quantityTransferred, $fromUser, $toUser));

    // 3. Ikiwa anayetuma ni Storekeeper, Admin apate notification
    if ($fromUser->isStorekeeper()) {
        $admins = User::where('role', 'admin')->get();
        // Admins wanapaswa kupokea "stock transferred" notification kutoka kwa storekeeper
        // Hapa tunaweza kutumia notification tofauti au kupitisha parameter ya ziada
        // Kwa sasa, tutatumia StockTransferredNotification kwa admins pia, lakini kwa message tofauti
        foreach ($admins as $admin) {
            $admin->notify(new StockTransferredNotification($product, $quantityTransferred, $fromUser, $toUser));
        }
    }
    // --- MWISHO WA NOTIFICATIONS ---

    return redirect()->route('stock-outputs.index')->with('success', 'Stock transferred successfully.');
}

    public function transferred()
    {
        // Hii inapaswa kuonyesha stock ambazo mtumiaji (seller) amepokea.
        // Kwa kuwa Products sasa zina user_id, tunaweza kutumia hiyo moja kwa moja.
        // Labda hii method haihitajiki sana kama seller dashboard tayari inaonyesha stock zake.
        $products = Product::where('user_id', Auth::id())->get();
        return view('products.transferred', compact('products'));
    }
 public function stockOverview(Request $request)
    {
        $search = $request->input('search');
        
        $query = Product::query();

        // Filter products based on user role to show only their owned stock
        if (Auth::user()->role === 'seller' || Auth::user()->role === 'storekeeper' || Auth::user()->role === 'admin') {
            $query->where('user_id', Auth::id());
        }
        
        if ($search) {
            $query->where('name', 'like', '%'.$search.'%');
        }

        // Get the paginated products first
        $products = $query->latest()->paginate(10); 

        // --- Calculate detailed quantities and profits for each product in the table ---
        // We use ->getCollection() because $products is a Paginator instance
        $products->getCollection()->transform(function ($product) {
            // Calculate input quantity for this specific product owned by the current user
            $product->input_quantity = (float) StockInput::where('product_id', $product->id)->sum('quantity');

            // Calculate output quantity for this specific product (transfers + sales by this user)
            $outputQuantityFromTransfers = (float) StockOutput::where('product_id', $product->id)
                                                    ->where('output_by_user_id', Auth::id())
                                                    ->sum('quantity');

            $salesForProduct = Sale::where('product_id', $product->id)
                                   ->where('sold_by_user_id', Auth::id())
                                   ->get(); // Get actual sales records to calculate profit per sale

            $outputQuantityFromSales = (float) $salesForProduct->sum('quantity');
            
            $product->output_quantity = $outputQuantityFromTransfers + $outputQuantityFromSales;

            // Remaining quantity is directly from the product model
            $product->remaining_quantity = (float) $product->quantity;

            // Profit per unit (for displaying in the table)
            $product->profit_per_unit = (float) ($product->selling_price_per_unit - $product->input_price_per_unit);

            // Total profit for THIS SPECIFIC PRODUCT based on its sales
            $productTotalProfit = 0;
            foreach ($salesForProduct as $sale) {
                // Ensure prices are numeric before calculation
                $sellingPrice = (float) $sale->selling_price_per_unit;
                $inputPrice = (float) $sale->input_price_per_unit;
                $quantitySold = (float) $sale->quantity;
                $productTotalProfit += ($sellingPrice - $inputPrice) * $quantitySold;
            }
            $product->total_profit = (float) $productTotalProfit;

            // Ensure price attributes for the table are correctly mapped and numeric
            $product->input_price = (float) $product->input_price_per_unit;
            $product->selling_price = (float) $product->selling_price_per_unit;

            return $product;
        });

        // --- Calculate overall summary totals (for the cards above the table) ---
        $userProductsForSummary = Product::where('user_id', Auth::id())->get(); // Get all products for summary

        $totalInputQuantitySummary = 0;
        $totalOutputQuantitySummary = 0;
        $totalRemainingQuantitySummary = 0;
        $totalSellingPriceOfCurrentStockSummary = 0;
        $totalProfitOfSoldItemsSummary = 0;

        foreach ($userProductsForSummary as $product) {
            $totalInputQuantitySummary += (float) StockInput::where('product_id', $product->id)->sum('quantity');

            $outputTransfers = (float) StockOutput::where('product_id', $product->id)
                                                    ->where('output_by_user_id', Auth::id())
                                                    ->sum('quantity');

            $salesForProductSummary = Sale::where('product_id', $product->id)
                                          ->where('sold_by_user_id', Auth::id())
                                          ->get();
            
            $outputSales = (float) $salesForProductSummary->sum('quantity');
            $totalOutputQuantitySummary += ($outputTransfers + $outputSales);
            
            foreach ($salesForProductSummary as $sale) {
                $totalProfitOfSoldItemsSummary += ((float) $sale->selling_price_per_unit - (float) $sale->input_price_per_unit) * (float) $sale->quantity;
            }
            
            $totalRemainingQuantitySummary += (float) $product->quantity;
            $totalSellingPriceOfCurrentStockSummary += (float) $product->quantity * (float) $product->selling_price_per_unit;
        }

        $totals = [
            'total_input' => $totalInputQuantitySummary,
            'total_output' => $totalOutputQuantitySummary,
            'total_remaining' => $totalRemainingQuantitySummary,
            'total_selling_price' => $totalSellingPriceOfCurrentStockSummary,
            'total_profit' => $totalProfitOfSoldItemsSummary,
        ];

        return view('products.overview', compact('products', 'totals'));
    }


public function show(Product $product)
{
    // Verify authorization
    if (Auth::user()->role === 'storekeeper' && $product->user_id != Auth::id()) {
        abort(403, 'Unauthorized action.');
    }

    $totalInput = $product->stockInputs->sum('quantity');
    $totalOutput = $product->stockOutputs->sum('quantity');
    
    return view('products.show', [
        'product' => $product,
        'totalInput' => $totalInput,
        'totalOutput' => $totalOutput,
        'remaining' => $product->quantity,
        'profitPerUnit' => $product->selling_price_per_unit - $product->input_price_per_unit,
        'totalProfit' => ($product->selling_price_per_unit - $product->input_price_per_unit) * $product->quantity
    ]);
}
}
