<?php

namespace App\Http\Controllers;

use App\Models\OtherIncome;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;
use Carbon\Carbon;
use App\Notifications\OtherIncomeRecordedNotification; // Import notification classes ulizounda
use App\Notifications\OtherIncomeDeletedNotification;

class OtherIncomeController extends Controller
{
    private $incomeSources = [
        'Rent Income',
        'Interest Income',
        'Dividend Income',
        'Sale of Assets',
        'Miscellaneous Income',
        'Other', // Kama "Others" ya expenses
    ];

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (Auth::user()->role === 'admin') {
            $otherIncomes = OtherIncome::with('user')->latest('income_date')->paginate(10);
        } else {
            $otherIncomes = OtherIncome::where('user_id', Auth::id())
                                        ->with('user')
                                        ->latest('income_date')
                                        ->paginate(10);
        }

        return view('other_incomes.index', compact('otherIncomes'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Admin tu ndiye anaruhusiwa kuongeza mapato mengine
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Huna ruhusa ya kuongeza mapato mengine.');
        }
        $incomeSources = $this->incomeSources;
        return view('other_incomes.create', compact('incomeSources'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Huna ruhusa ya kuhifadhi mapato mengine.');
        }

        $rules = [
            'amount' => 'required|numeric|min:0.01',
            'income_date' => 'required|date',
            'source' => 'required|string|in:' . implode(',', $this->incomeSources),
            'description' => 'nullable|string|max:255',
        ];

        // Kama chanzo ni 'Other', basi 'description' ni lazima
        if ($request->source === 'Other') {
            $rules['description'] = 'required|string|max:255';
        }

        $request->validate($rules);

        OtherIncome::create([
            'user_id' => Auth::id(), // Rekodi mtumiaji aliyeingiza mapato
            'amount' => $request->amount,
            'income_date' => $request->income_date,
            'source' => $request->source,
            'description' => ($request->source === 'Other') ? $request->description : $request->description, // Description inaweza kuwa maelezo ya ziada au maelezo ya 'Other'
        ]);

         // --- NOTIFICATIONS HAPA ---
    $loggedInUser = Auth::user();

    // Tuma notification kwa admin wote (ikiwemo anayerekodi)
    $admins = User::where('role', 'admin')->get();
    Notification::send($admins, new OtherIncomeRecordedNotification($otherIncome, $loggedInUser));
    // --- MWISHO WA NOTIFICATIONS ---


        return redirect()->route('other-incomes.index')->with('success', 'Mapato mengine yameongezwa kikamilifu.');
    }

    /**
     * Display the specified resource.
     */
    public function show(OtherIncome $otherIncome)
    {
        if (Auth::user()->role !== 'admin' && $otherIncome->user_id !== Auth::id()) {
            abort(403, 'Huna ruhusa ya kuona rekodi hii.');
        }
        return view('other_incomes.show', compact('otherIncome'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(OtherIncome $otherIncome)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Huna ruhusa ya kuhariri rekodi hii.');
        }
        $incomeSources = $this->incomeSources;
        return view('other_incomes.edit', compact('otherIncome', 'incomeSources'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, OtherIncome $otherIncome)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Huna ruhusa ya kusasisha rekodi hii.');
        }

        $rules = [
            'amount' => 'required|numeric|min:0.01',
            'income_date' => 'required|date',
            'source' => 'required|string|in:' . implode(',', $this->incomeSources),
            'description' => 'nullable|string|max:255',
        ];

        if ($request->source === 'Other') {
            $rules['description'] = 'required|string|max:255';
        }

        $request->validate($rules);

        $otherIncome->update([
            'amount' => $request->amount,
            'income_date' => $request->income_date,
            'source' => $request->source,
            'description' => ($request->source === 'Other') ? $request->description : $request->description,
        ]);

        return redirect()->route('other-incomes.index')->with('success', 'Mapato mengine yamesasishwa kikamilifu.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(OtherIncome $otherIncome)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Huna ruhusa ya kufuta rekodi hii.');
        }
        
        $otherIncome->delete();

          // --- NOTIFICATIONS HAPA ---
    // Tuma notification kwa admin wote
    $admins = User::where('role', 'admin')->get();
    Notification::send($admins, new OtherIncomeDeletedNotification($incomeDetails, $deletedByUser));
    // --- MWISHO WA NOTIFICATIONS ---


        return redirect()->route('other-incomes.index')->with('success', 'Mapato mengine yamefutwa kikamilifu.');
    }
}