<?php

namespace App\Http\Controllers;

use App\Models\Expense;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification; // MUHIMU: Hii ni kwa ajili ya kutuma notifications
use App\Notifications\ExpenseRecordedNotification; // Import notification classes ulizounda
use App\Notifications\ExpenseDeletedNotification;

class ExpenseController extends Controller
{
    // Orodha ya kategoria za matumizi
    private $expenseCategories = [
        'Carriage Inwards',
        'Carriage Outwards',
        'Accounting expense',
        'Advertising',
        'Amortization',
        'Bad Debts',
        'Provision for Doubtful Debts',
        'Employee Payroll Tax',
        'Entertainment',
        'Interest Expenses',
        'Miscellaneous',
        'Software',
        'Telephone Bill',
        'Internet Bill',
        'Web Hosting',
        'Vehicle Expenses',
        'Maintenance and Repair',
        'Travelling',
        'Meeting',
        'Utilities',
        'Suppliers and Consumable Expenses',
        'Water Bill',
        'Rent Bill',
        'Salaries and Wages',
        'Discount Allowed',
        'Others',
    ];

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (Auth::user()->role === 'admin') {
            $expenses = Expense::with('user')->latest('expense_date')->paginate(10);
        } else {
            $expenses = Expense::where('user_id', Auth::id())
                ->with('user')
                ->latest('expense_date')
                ->paginate(10);
        }

        return view('expenses.index', compact('expenses'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $expenseCategories = $this->expenseCategories; // Tumia array yetu
        return view('expenses.create', compact('expenseCategories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $rules = [
            'category_name' => 'required|string|in:' . implode(',', $this->expenseCategories), // Kategoria ni lazima na iwe kwenye orodha
            'amount' => 'required|numeric|min:0.01',
            'expense_date' => 'required|date',
            'description' => 'nullable|string|max:255', // Description ya ziada, si lazima
        ];

        // Kama kategoria iliyochaguliwa ni 'Others', basi 'other_details' ni lazima
        if ($request->category_name === 'Others') {
            $rules['other_details'] = 'required|string|max:255';
        }

        $request->validate($rules);

        // Unda array ya data ya kuhifadhi
        $dataToStore = [
            'category_name' => $request->category_name,
            'amount' => $request->amount,
            'expense_date' => $request->expense_date,
            'user_id' => Auth::id(),
            'description' => $request->description, // Hii sasa inahifadhi maelezo ya ziada (optional)
            'other_details' => ($request->category_name === 'Others') ? $request->other_details : null, // Weka 'other_details' tu kama kategoria ni 'Others'
        ];

        // Hifadhi Expense na hifadhi object iliyoundwa kwenye $expense variable
        $expense = Expense::create($dataToStore); // <--- HII NDIYO MAREKEBISHO MUHIMU HAPA!

        // Tuma notifications
        $loggedInUser = Auth::user();
        $loggedInUser->notify(new ExpenseRecordedNotification($expense, $loggedInUser));

        if ($loggedInUser->isAdmin() || $loggedInUser->isStorekeeper()) {
            $admins = User::where('role', 'admin')->get();
            // Tuma notification kwa admins wengine, ukiondoa yule aliyelog in
            Notification::send($admins->except($loggedInUser->id), new ExpenseRecordedNotification($expense, $loggedInUser));
        }

        return redirect()->route('expenses.index')->with('success', 'Matumizi yameongezwa kikamilifu.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Expense $expense)
    {
        if (Auth::user()->role !== 'admin' && $expense->user_id !== Auth::id()) {
            abort(403, 'Huna ruhusa ya kuona matumizi haya.');
        }
        return view('expenses.show', compact('expense'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Expense $expense)
    {
        if (Auth::user()->role !== 'admin' && $expense->user_id !== Auth::id()) {
            abort(403, 'Huna ruhusa ya kuhariri matumizi haya.');
        }
        $expenseCategories = $this->expenseCategories; // Tumia array yetu
        return view('expenses.edit', compact('expense', 'expenseCategories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Expense $expense)
    {
        if (Auth::user()->role !== 'admin' && $expense->user_id !== Auth::id()) {
            abort(403, 'Huna ruhusa ya kuhariri matumizi haya.');
        }

        $rules = [
            'category_name' => 'required|string|in:' . implode(',', $this->expenseCategories),
            'amount' => 'required|numeric|min:0.01',
            'expense_date' => 'required|date',
            'description' => 'nullable|string|max:255',
        ];

        if ($request->category_name === 'Others') {
            $rules['other_details'] = 'required|string|max:255';
        }

        $request->validate($rules);

        // Unda array ya data ya kusasisha
        $dataToUpdate = [
            'category_name' => $request->category_name,
            'amount' => $request->amount,
            'expense_date' => $request->expense_date,
            'description' => $request->description,
            'other_details' => ($request->category_name === 'Others') ? $request->other_details : null,
        ];

        $expense->update($dataToUpdate);

        return redirect()->route('expenses.index')->with('success', 'Matumizi yamebadilishwa kikamilifu.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Expense $expense)
    {
        if (Auth::user()->role !== 'admin' && $expense->user_id !== Auth::id()) {
            abort(403, 'Huna ruhusa ya kufuta matumizi haya.');
        }

        // Hifadhi details kabla ya kufuta $expense
        $expenseDetails = "Expense: {$expense->category_name} - TZS " . number_format($expense->amount, 2);
        $deletedByUser = Auth::user();

        $expense->delete(); // Futa expense

        // Tuma notifications
        $deletedByUser->notify(new ExpenseDeletedNotification($expenseDetails, $deletedByUser));

        $admins = User::where('role', 'admin')->get();
        // Tuma notification kwa admins wengine, ukiondoa yule aliyefuta
        Notification::send($admins->except($deletedByUser->id), new ExpenseDeletedNotification($expenseDetails, $deletedByUser));

        return redirect()->route('expenses.index')->with('success', 'Matumizi yamefutwa kikamilifu.');
    }
}