<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User; 
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class CreateAdminUser extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'make:admin {name?} {email?} {password?}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Creates a new admin user';

    /**
     * Execute the console command.
     */
    public function handle()
    {
     
        $name = $this->argument('name') ?? $this->ask('Enter the full name of the admin');
        $email = $this->argument('email') ?? $this->ask('Enter the admin\'s email address');
        $password = $this->argument('password') ?? $this->secret('Enter the admin\'s password');
        $password_confirmation = $this->secret('Repeat the password');

        if ($password !== $password_confirmation) {
            $this->error('Passwords do not match. Please try again.');
            return 1; 
        }

    
        $validator = Validator::make([
            'name' => $name,
            'email' => $email,
            'password' => $password,
        ], [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8'],
        ]);

        if ($validator->fails()) {
            foreach ($validator->errors()->all() as $error) {
                $this->error($error);
            }
            return 1; 
        }

        try {
            
            $admin = User::create([
                'name' => $name,
                'email' => $email,
                'password' => Hash::make($password),
                'role' => 'admin', 
            ]);

            

            $this->info("Admin user {$admin->name} created successfully!");
            $this->info("Email: {$admin->email}");
            $this->info("Password: The one you entered (not displayed for security)");

        } catch (\Exception $e) {
            $this->error('An error occurred while creating the admin user: ' . $e->getMessage());
            return 1; 
        }

        return 0; 
    }
}
